/*
Module : SocServer.cpp
Purpose: A simple sockets server which implements the "time" protocol RFC 868 using
         the socket wrapper classes provided 
Created: PJN / 20-08-2004
History: PJN / 27-01-2007 1. Updated copyright details
                          2. Updated the code to clean compile on VC 2005
         PJN / 09-01-2011 1. Updated to support the IPv6 functionality exposed by CWSocket class
         PJN / 24-07-2016 1. Updated copyright details.
                          2. Updated code to compile cleanly in VC 2010 to VC 2015
         PJN / 13-08-2017 1. Updated copyright details.
                          2. Replaced CString::operator LPC*STR() calls throughout the codebase with
                          CString::GetString calls
         PJN / 01-01-2019 1. Updated copyright details.
                          2. Fixed a number of C++ core guidelines compiler warnings. These changes
                          mean that the code will now only compile on VC 2017 or later.
         PJN / 28-09-2019 1. Fixed a number of compiler warnings when the code is compiled with 
                          VS 2019 Preview
         PJN / 06-12-2020 1. Fixed Clang-Tidy static code analysis warnings in the code.
         PJN / 28-05-2022 1. Updated copyright details.
                          2. Updated the code to use C++ uniform initialization for all variable declarations.

Copyright (c) 2004 - 2022 by PJ Naughter (Web: www.naughter.com, Email: pjna@naughter.com)

All rights reserved.

Copyright / Usage Details:

You are allowed to include the source code in any product (commercial, shareware, freeware or otherwise) 
when your product is released in binary form. You are allowed to modify the source code in any way you want 
except you cannot modify the copyright details at the top of each module. If you want to distribute source 
code with your application, then you are only allowed to distribute versions released by the author. This is 
to maintain a single distribution point for the source code.

*/


//////////////////// Includes /////////////////////////////////////////////////

#include "stdafx.h"
#include "SocMFC.h"


//////////////////// Macros / Defines /////////////////////////////////////////

#ifdef _DEBUG
#define new DEBUG_NEW
#endif

constexpr const int g_nPort{37};


//////////////////// Implementation ///////////////////////////////////////////

#pragma warning(suppress: 26426)
CWinApp theApp;

int _tmain()
{
  //initialize MFC (Note we must put up with some /analyze warnings for AfxWinInit even in VC 2008)
  if (!AfxWinInit(::GetModuleHandle(nullptr), nullptr, ::GetCommandLine(), 0))
  {
    _tprintf(_T("Failed to initialize MFC\n"));
    return 1;
  }

  //Initialise Sockets
  WSADATA wsaData{};
  if (WSAStartup(MAKEWORD(2, 2), &wsaData) != 0)
  {
    _tprintf(_T("Failed to initialise Winsock, GetLastError:%u\n"), GetLastError()); //NOLINT(clang-diagnostic-format)
    return FALSE;
  }

  //Bind and listen
  CWSocket serverSocket;
  try
  {
    //Bind to the port to listen on
    serverSocket.CreateAndBind(g_nPort);

    //Start listening for client connections
    serverSocket.Listen();

    //Sit in a loop accepting client connections, a real app would use some termination
    //condtion to break out of this loop. Also in fact this whole slice of code
    //would probably be implemented in a worker thread and would use event notifications
    //to be informed when client connections are accepted
    while (true)
    {
      //Accept a client connection
      CWSocket clientSocket;
      SOCKADDR_INET clientAddress;
      memset(&clientAddress, 0, sizeof(clientAddress));
      int nAddressLength{sizeof(clientAddress)};
#pragma warning(suppress: 26486 26490)
      serverSocket.Accept(clientSocket, reinterpret_cast<SOCKADDR*>(&clientAddress), &nAddressLength);

      //Send the data down the connection (Note that in a real world example, the
      //actual clientSocket operations would probably be implemented in a worker thread
      //or via a thread pool). This would allows the code to immediately loop around and
      //accept another client connection. This simple example can only handle one client
      //at a time due to its single threaded implementation
      time_t t{0};
      time(&t);
#pragma warning(suppress: 26489)
      CWSocket::String sPeerAddress;
      UINT nPeerPort{0};
      clientSocket.GetPeerName(sPeerAddress, nPeerPort);
    #ifdef CWSOCKET_MFC_EXTENSIONS
      _tprintf(_T("Handling client request from %s:%u\n"), sPeerAddress.GetString(), nPeerPort);
    #else
      _tprintf(_T("Handling client request from %s:%u\n"), sPeerAddress.c_str(), nPeerPort);
    #endif //#ifdef CWSOCKET_MFC_EXTENSIONS
      clientSocket.Send(&t, sizeof(t));

      SOCKADDR_STORAGE demo{};
      nAddressLength = sizeof(demo);
#pragma warning(suppress: 26490)
      clientSocket.GetPeerName(reinterpret_cast<SOCKADDR*>(&demo), &nAddressLength);
      nAddressLength = sizeof(demo);
#pragma warning(suppress: 26490)
      clientSocket.GetSockName(reinterpret_cast<SOCKADDR*>(&demo), &nAddressLength);
    }
  }
#ifdef CWSOCKET_MFC_EXTENSIONS
  catch(CWSocketException* pEx)
  {
    _tprintf(_T("Unknown socket exception: Error:%d\n"), pEx->m_nError);
    pEx->Delete();
    WSACleanup();
    return 3;
  }
#else
#pragma warning(suppress: 26496)
  catch (CWSocketException& e)
  {
    _tprintf(_T("Unknown socket exception: Error:%d\n"), e.m_nError);
    WSACleanup();
    return 3;
  }
#endif //#ifdef CWSOCKET_MFC_EXTENSIONS

  WSACleanup();
  return 0;
}
