//////////////////////////////// Includes /////////////////////////////////////

#include "stdafx.h"
#include "HTTPServerWrappers.h"
#include "CSecHandle.h"
#include "CCredHandle.h"
#include "CPPCThreadPool.h"


//////////////////////////////// Implementation ///////////////////////////////

std::thread::id test_function(size_t delay)
{
  std::this_thread::sleep_for(std::chrono::milliseconds(delay + 1));
  return std::this_thread::get_id();
}

constexpr int sum(int a, int b)
{
  return a + b;
}

void throwRuntimeError()
{
  throw std::runtime_error{"Error"};
}

constexpr void accumulate_with_refs(int nValue, const int& nAddition, int& nResult)
{
  nResult = nValue + nAddition;
}

#pragma warning(suppress: 26429)
std::thread::id test_submitfromworkerthread(CppConcurrency::CThreadPool* pPool, size_t delay)
{
  //Validate our parameters
  ATLASSERT(pPool != nullptr);

  _tprintf(_T(" test_submitfromworkerthread, Processing item\n"));
  if ((rand() % 4) == 0)
  {
    _tprintf(_T(" test_submitfromworkerthread, Resubmitting first item from worker thread\n"));
    pPool->submit(test_submitfromworkerthread, pPool, 10);
  }
  std::this_thread::sleep_for(std::chrono::milliseconds(delay + 1));
  if ((rand() % 4) == 0)
  {
    _tprintf(_T(" test_submitfromworkerthread, Resubmitting second item from worker thread\n"));
    pPool->submit(test_submitfromworkerthread, pPool, 10);
  }
  return std::this_thread::get_id();
}

class CThreadPoolWithCustomInitInstanceThread : public CppConcurrency::CThreadPool
{
public:
//Constructors / Destructors
  CThreadPoolWithCustomInitInstanceThread(size_t thread_count) : CThreadPool{thread_count}
  {
  }
  CThreadPoolWithCustomInitInstanceThread(const CThreadPoolWithCustomInitInstanceThread&) = delete;
  CThreadPoolWithCustomInitInstanceThread(CThreadPoolWithCustomInitInstanceThread&&) = delete;
  ~CThreadPoolWithCustomInitInstanceThread() = default; //NOLINT(modernize-use-override)

//Methods
  CThreadPoolWithCustomInitInstanceThread& operator=(const CThreadPoolWithCustomInitInstanceThread&) = delete;
  CThreadPoolWithCustomInitInstanceThread& operator=(CThreadPoolWithCustomInitInstanceThread&&) = delete;

//Member variables
  std::atomic<int> _ThreadsWhichRanInitInstanceThread{0};

protected:
//Methods
  bool InitInstanceThread(size_t threadIndex) override
  {
    ++_ThreadsWhichRanInitInstanceThread;
    return __super::InitInstanceThread(threadIndex);
  }
};

class CThreadPoolWithStartupFailures : public CppConcurrency::CThreadPool
{
public:
//Constructors / Destructors
  CThreadPoolWithStartupFailures(size_t thread_count) : CThreadPool{thread_count}
  {
  }
  CThreadPoolWithStartupFailures(const CThreadPoolWithStartupFailures&) = delete;
  CThreadPoolWithStartupFailures(CThreadPoolWithStartupFailures&&) = delete;
  ~CThreadPoolWithStartupFailures() = default; //NOLINT(modernize-use-override)

//Methods
  CThreadPoolWithStartupFailures& operator=(const CThreadPoolWithStartupFailures&) = delete;
  CThreadPoolWithStartupFailures& operator=(CThreadPoolWithStartupFailures&&) = delete;

protected:
//Methods
  bool InitInstanceThread(size_t threadIndex) noexcept override
  {
    //For test purposes only allow a thread pool thread with index 0 to start
    //successfully. This will cause the unit tests with thread pools using this
    //class with more threads than 1 in the pool to fail their calls to "WaitForInitInstance"
    return threadIndex == 0;
  }
};


class CThreadPoolWithCustomeWorkerThread : public CppConcurrency::CThreadPool
{
public:
  //Constructors / Destructors
  CThreadPoolWithCustomeWorkerThread(unsigned thread_count) : CThreadPool{thread_count}
  {
  }
  CThreadPoolWithCustomeWorkerThread(const CThreadPoolWithCustomeWorkerThread&) = delete;
  CThreadPoolWithCustomeWorkerThread(CThreadPoolWithCustomeWorkerThread&&) = delete;
  ~CThreadPoolWithCustomeWorkerThread() = default; //NOLINT(modernize-use-override)

  //Methods
  CThreadPoolWithCustomeWorkerThread& operator=(const CThreadPoolWithCustomeWorkerThread&) = delete;
  CThreadPoolWithCustomeWorkerThread& operator=(CThreadPoolWithCustomeWorkerThread&&) = delete;

protected:
//Methods
  void WorkerThread(size_t nThreadIndex) override
  {
    constexpr int nX = 42;
    UNREFERENCED_PARAMETER(nX);
    __super::WorkerThread(nThreadIndex);
  }
};


class CSampleThreadPool : public CppConcurrency::CThreadPool
{
public:
//Constructors / Destructors
#ifdef _MSC_VER
#pragma warning(suppress: 26455)
#endif //#ifdef _MSC_VER
  CSampleThreadPool(_In_ size_t threadCount = 0,
                    _In_ bool permitTaskStealing = true,
                    _In_ bool paused = false,
                    _In_ bool pumpWindowsMessages = false) : CppConcurrency::CThreadPool{threadCount, permitTaskStealing, paused, pumpWindowsMessages}
  {
  }

//Methods
  //Lets override one method to see it being called
  bool InitInstanceThread(_In_ size_t /*threadIndex*/) noexcept override
  {
    return true;
  }
};


class CSampleHTTPServer : public HTTPServer::CServer<CSampleThreadPool>
{
public:
//Constructors / Destructors
  CSampleHTTPServer(_In_ size_t nThreadCount = 0,
                    _In_ bool bPermitTaskStealing = true,
                    _In_ bool bPaused = false,
                    _In_ bool bPumpWindowsMessages = false) noexcept : HTTPServer::CServer<CSampleThreadPool>{nThreadCount, bPermitTaskStealing, bPaused, bPumpWindowsMessages},
                                                                       m_NTLMSS{SEC_E_OK},
                                                                       m_cbMaxNTLMTokenSize{0},
                                                                       m_NegotiateSS{SEC_E_OK},
                                                                       m_cbMaxNegotiateTokenSize{0}
  {
    //Create the NTLM credentials handle
    TimeStamp tsExpires{};
#ifdef _UNICODE
#pragma warning(suppress: 26465 26492)
    m_NTLMSS = AcquireCredentialsHandle(nullptr, const_cast<LPWSTR>(NTLMSP_NAME), SECPKG_CRED_INBOUND, nullptr, nullptr, nullptr, nullptr, &m_NTLMCred.m_Handle, &tsExpires);
#else
#pragma warning(suppress: 26465 26492)
    m_NTLMSS = AcquireCredentialsHandle(nullptr, const_cast<LPSTR>(NTLMSP_NAME_A), SECPKG_CRED_INBOUND, nullptr, nullptr, nullptr, nullptr, &m_NTLMCred.m_Handle, &tsExpires);
#endif //#ifdef _UNICODE

    //Also get the NTLM max token size
    if (m_NTLMSS == SEC_E_OK)
    {
      PSecPkgInfo pSecInfo{nullptr};
  #ifdef _UNICODE
  #pragma warning(suppress: 26465 26492)
      m_NTLMSS = QuerySecurityPackageInfo(const_cast<LPTSTR>(NTLMSP_NAME), &pSecInfo);
  #else
  #pragma warning(suppress: 26465 26492)
      m_NTLMSS = QuerySecurityPackageInfo(const_cast<LPTSTR>(NTLMSP_NAME_A), &pSecInfo);
  #endif //#ifdef _UNICODE
      if (m_NTLMSS == SEC_E_OK)
      {
        m_cbMaxNTLMTokenSize = pSecInfo->cbMaxToken;
        FreeContextBuffer(pSecInfo);
      }
    }

    //Create the Negotiate credentials handle
#pragma warning(suppress: 26465 26492)
    m_NegotiateSS = AcquireCredentialsHandle(nullptr, const_cast<LPTSTR>(NEGOSSP_NAME), SECPKG_CRED_INBOUND, nullptr, nullptr, nullptr, nullptr, &m_NegotiateCred.m_Handle, &tsExpires);

    //Also get the Negotiate max token size
    if (m_NegotiateSS == SEC_E_OK)
    {
      PSecPkgInfo pSecInfo{nullptr};
#pragma warning(suppress: 26465 26492)
      m_NegotiateSS = QuerySecurityPackageInfo(const_cast<LPTSTR>(NEGOSSP_NAME), &pSecInfo);
      if (m_NegotiateSS == SEC_E_OK)
      {
        m_cbMaxNegotiateTokenSize = pSecInfo->cbMaxToken;
        FreeContextBuffer(pSecInfo);
      }
    }
  }
  CSampleHTTPServer(const CSampleHTTPServer&) = delete;
  CSampleHTTPServer(CSampleHTTPServer&&) = delete;
  ~CSampleHTTPServer()= default; //NOLINT(modernize-use-override)

//Methods
  CSampleHTTPServer& operator=(const CSampleHTTPServer&) = delete;
  CSampleHTTPServer& operator=(CSampleHTTPServer&&) = delete;

protected:
//Internal session class used to allow us to handle negotiate & digest credentials in follow on request and cookie sessions
  class CSession
  {
  public:
  //Member variables
    std::shared_ptr<CSecHandle> m_Context;
    ULONGLONG m_dwStartTickCount = 0; //The tick count time when this session was created
    CStringA m_sNonce; //The last "nonce" value the server issued for this session
    CString m_sUsername; //The username associated with this session
    CStringA m_sPasswordHash; //The MD5 hash of the password for this session
    bool m_bPersistentSession = false; //Should this session be exempt from killing
  };

//Methods

/// @brief Creates a session id from a GUID suitable for sending to the client as a cookie
  static HRESULT CreateSessionID(_Inout_ CStringA& sSessionID)
  {
    UUID uuid{};
    RPC_STATUS status{UuidCreate(&uuid)};
    if ((status != RPC_S_OK) && (status != RPC_S_UUID_LOCAL_ONLY))
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_RPC, status);
    RPC_CSTR pszGuid{nullptr};
    status = UuidToStringA(&uuid, &pszGuid);
    if (status != RPC_S_OK)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_RPC, status);
#pragma warning(suppress: 26490)
    sSessionID = reinterpret_cast<char*>(pszGuid);
    RpcStringFreeA(&pszGuid);
    sSessionID.Remove('-');
    return S_OK;
  }

/// @brief An example which implements a simple directory listing. The files returned are returned
/// from the current working directory. Note that this would not be good security practice for a 
/// real production web server!
  HRESULT HandleDirectoryListing(_In_ const HTTP_REQUEST& request)
  {
    WIN32_FIND_DATA ffd{};
    HANDLE hFind{FindFirstFile(_T("*.*"), &ffd)};
    if (hFind == INVALID_HANDLE_VALUE)
    {
      const DWORD dwError{GetLastError()};
      SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, dwError);
    }

    //List all the files in the directory with a URL to each one
    CStringA sBody{R"(<html><head><meta charset="UTF-8"><title>Demo directory listing for HTTPServerWrappers</title></head><body><table>)"};
    do
    {
      if ((ffd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) == 0)
      {
        //Form the URL of the file
#pragma warning(suppress: 26485)
        ATL::CT2A sFilename{ffd.cFileName, CP_UTF8};
        CStringW sURL{request.CookedUrl.pAbsPath};
        const int nQueryStringOffset{sURL.ReverseFind(L'?')};
        sURL = sURL.Mid(0, nQueryStringOffset);
        sURL += L"/";
        sURL += sFilename;
        sURL += L"?file";

        //Get the last modified date and time as strings
        CStringA sDate;
        CStringA sTod;
        SYSTEMTIME st{};
        if (FileTimeToSystemTime(&ffd.ftLastWriteTime, &st))
        {
          GetDateFormatA(LOCALE_SYSTEM_DEFAULT, LOCALE_NOUSEROVERRIDE, &st, nullptr, sDate.GetBuffer(128), 128);
          sDate.ReleaseBuffer();
          GetTimeFormatA(LOCALE_SYSTEM_DEFAULT, LOCALE_NOUSEROVERRIDE, &st, nullptr, sTod.GetBuffer(128), 128);
          sTod.ReleaseBuffer();

          //Form all the info into a row of the table
          ATL::CW2A sUtf8URL{sURL, CP_UTF8};
          sBody.AppendFormat("<tr>\r\n<td><a href=%s>%s</a></td><td>%I64uKB</td><td>%s</td><td>%s</td>\r\n</tr>",
                             sUtf8URL.operator LPSTR(), sFilename.operator LPSTR(), ((static_cast<unsigned __int64>(ffd.nFileSizeHigh) << 32) + static_cast<unsigned __int64>(ffd.nFileSizeLow) + 1023) / 1024, sDate.GetString(), sTod.GetString());
        }
      }
    }
    while (FindNextFile(hFind, &ffd) != 0);
    FindClose(hFind);
    sBody += R"(</table></body></html>)";

    //Get the current system time in UTC
    SYSTEMTIME stCurTime{};
    ::GetSystemTime(&stCurTime);

    //Initialize the HTTP response structure
    HTTP_RESPONSE response{};
    InitializeHTTPResponse(response, 200, "OK");

    //Add the entity string to the response if one was provided
    HTTP_DATA_CHUNK dataChunk{};
#pragma warning(suppress: 6387)
    AddKnownHeader(response, HttpHeaderContentType, "text/html");

    //Add an entity chunk
    dataChunk.DataChunkType = HttpDataChunkFromMemory;
#pragma warning(suppress: 26492)
    dataChunk.FromMemory.pBuffer = const_cast<LPSTR>(sBody.GetString());
#pragma warning(suppress: 26472)
    dataChunk.FromMemory.BufferLength = sBody.GetLength();
    response.EntityChunkCount = 1;
    response.pEntityChunks = &dataChunk;

    //Because the entity body is sent in one call, it is not required to specify the Content-Length
    ULONG nBytesSent = 0;
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
  }

/// @brief An example which implements returning the contents of a file. Note that there is no robust validation
/// done to ensure the file returned is from a specific server controlled directory. This would not be good security
/// practice for a real production web server! Most of the work is implemented by HTTPServer::CServer::SendFileResponse.
  HRESULT HandleFileRequest(_In_ const HTTP_REQUEST& request)
  {
    //Extract the name of the file to serve back from the URL
    CStringW sFile{request.CookedUrl.pAbsPath};
    const int nQueryStringOffset{sFile.ReverseFind(L'?')};
    if (nQueryStringOffset == -1)
    {
      SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      return E_FAIL;
    }
    sFile = sFile.Mid(0, nQueryStringOffset);
    const int nFinalSlash{sFile.ReverseFind(L'/')};
    if (nFinalSlash == -1)
    {
      SendResponse(request.RequestId, 404, "Not found", nullptr, nullptr, 0);
      return E_FAIL;
    }
    sFile = sFile.Mid(nFinalSlash + 1);
    return SendFileResponse(request, CString(sFile), "application/octet-stream", true);
  }

/// @brief An example which demonstrates a POST request which echo's back the entity from the request into the response
  HRESULT HandleEchoPost(_In_ const HTTP_REQUEST& request)
  {
    //Check to see if the entity body exists in the request already. This would be the case if the HTTP_RECEIVE_REQUEST_FLAG_COPY_BODY flag 
    //was used in the HttpReceiveHttpRequest call. In that case we could simply use the HTTP_REQUEST::pEntityChunks field directly
    //rather than what we are doing here which is explicitly reading the entity a chunk at a time from the client
    if (request.Flags & HTTP_REQUEST_FLAG_MORE_ENTITY_BODY_EXISTS)
    {
      CString sTempPath;
      const DWORD dwGTP{GetTempPath(_MAX_PATH, sTempPath.GetBuffer(_MAX_PATH))};
      sTempPath.ReleaseBuffer();
      if (dwGTP == 0)
      {
        const DWORD dwError{GetLastError()};
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, dwError);
      }

      //The entity body is sent over multiple calls. Collect these in a temporary file and send back
      CString sTempFile;
      const UINT nGTFN{GetTempFileName(sTempPath, _T("New"), 0, sTempFile.GetBuffer(_MAX_PATH))};
      sTempFile.ReleaseBuffer();
      if (nGTFN == 0)
      {
        const DWORD dwError{GetLastError()};
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, dwError);
      }
      ATL::CAtlFile tempFile;
      HRESULT hr{tempFile.Create(sTempFile, GENERIC_READ | GENERIC_WRITE, 0, CREATE_ALWAYS)};
      if (FAILED(hr))
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return hr;
      }

      //Allocate space for reading the entity buffer
      constexpr ULONG nEntityBufferLength{2048};
      std::vector<BYTE> entityBuffer{nEntityBufferLength, std::allocator<BYTE>{}};
      auto pEntityBuffer{entityBuffer.data()};

      //Read all the entity body from the HTTP request and echo it back in the HTTP response
      unsigned __int64 nTotalBytesRead{0};
      do
      {
        //Read the entity chunk from the request
        ULONG nBytesRead{0};
        ULONG nResult{m_Queue.ReceiveRequestEntityBody(request.RequestId, 0, pEntityBuffer, nEntityBufferLength, &nBytesRead, nullptr)};
        switch (nResult)
        {
          case NO_ERROR:
          {
            if (nBytesRead != 0)
            {
              nTotalBytesRead += nBytesRead;
              hr = tempFile.Write(pEntityBuffer, nBytesRead);
              if (FAILED(hr))
              {
                tempFile.Close();
                DeleteFile(sTempFile);
                SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
                return hr;
              }
            }
            break;
          }
          case ERROR_HANDLE_EOF:
          {
            //The last request entity body has been read. Send back a response
            //To illustrate entity sends via HttpSendResponseEntityBody, the response will
            //be sent over multiple calls. To do this, pass the HTTP_SEND_RESPONSE_FLAG_MORE_DATA flag.
            if (nBytesRead != 0)
            {
              nTotalBytesRead += nBytesRead;
              hr = tempFile.Write(pEntityBuffer, nBytesRead);
              if (FAILED(hr))
              {
                tempFile.Close();
                DeleteFile(sTempFile);
                SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
                return hr;
              }
            }

            //Initialize the HTTP response structure
            HTTP_RESPONSE response{};
            InitializeHTTPResponse(response, 200, "OK");

            //Because the response is sent over multiple API calls, add a content length header
            CStringA sContentLength;
            sContentLength.Format("%I64u", nTotalBytesRead);
            AddKnownHeader(response, HttpHeaderContentLength, sContentLength.GetString());

            ULONG nBytesSent{0};
            nResult = m_Queue.SendResponse(request.RequestId, HTTP_SEND_RESPONSE_FLAG_MORE_DATA, &response, nullptr, &nBytesSent, nullptr, nullptr);
            if (nResult != NO_ERROR)
            {
              tempFile.Close();
              DeleteFile(sTempFile);
              SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
              return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);
            }

            //Send entity body from a file handle
            HTTP_DATA_CHUNK dataChunk{};
            dataChunk.DataChunkType = HttpDataChunkFromFileHandle;
            dataChunk.FromFileHandle.ByteRange.StartingOffset.QuadPart = 0;
            dataChunk.FromFileHandle.ByteRange.Length.QuadPart = HTTP_BYTE_RANGE_TO_EOF;
            dataChunk.FromFileHandle.FileHandle = tempFile;
            nResult = m_Queue.SendEntityBody(request.RequestId, 0, 1, &dataChunk, nullptr, nullptr, nullptr);
            return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);
            break;
          }
          default:
          {
            tempFile.Close();
            DeleteFile(sTempFile);
            SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
            return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);
          }
        }
      }
      while (true);
    }
    else
    {
      //Initialize the HTTP response structure
      HTTP_RESPONSE response{};
      InitializeHTTPResponse(response, 200, "OK");

      //If the response will be sent over multiple API calls then add a content length header
      if (request.EntityChunkCount)
      {
        CStringA sContentLength;
        sContentLength.Format("%u", request.pEntityChunks->FromMemory.BufferLength);
        AddKnownHeader(response, HttpHeaderContentLength, sContentLength.GetString());
      }

      //This request does not have an entity body
      ULONG nBytesSent{0};
      ULONG nResult{m_Queue.SendResponse(request.RequestId, request.EntityChunkCount ? HTTP_SEND_RESPONSE_FLAG_MORE_DATA : 0, &response, nullptr, &nBytesSent, nullptr, nullptr)};
      if (nResult != NO_ERROR)
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);

      //Echo entity chunks from the request
      if (request.EntityChunkCount)
      {
        HTTP_DATA_CHUNK dataChunk{};
        dataChunk.DataChunkType = HttpDataChunkFromMemory;
        dataChunk.FromMemory.BufferLength = request.pEntityChunks->FromMemory.BufferLength;
        dataChunk.FromMemory.pBuffer = request.pEntityChunks->FromMemory.pBuffer;
        nResult = m_Queue.SendEntityBody(request.RequestId, 0, 1, &dataChunk, nullptr, nullptr, nullptr);
      }
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);
    }
  }

/// @brief Method which allows HTTP binary data to be printed for demonstration purposes
  bool AuditData(_In_opt_ LPCTSTR pszTitle, _In_reads_bytes_(lSize) const BYTE* pbyData, _In_ ULONG lSize)
  {
    LPTSTR pszString{nullptr};
    std::vector<TCHAR> auditString;
    if (lSize != 0)
    {
      //First call to get the length of the buffer to allocate
      DWORD cchString = 0;
      if (!CryptBinaryToString(pbyData, lSize, CRYPT_STRING_HEXASCII, nullptr, &cchString))
        return false;

      //Allocate enough memory and recall
      auditString.resize(cchString);
      if (!CryptBinaryToString(pbyData, lSize, CRYPT_STRING_HEXASCII, auditString.data(), &cchString))
        return false;
      pszString = auditString.data();
    }

    //Audit the data using printf
    CString sData;
    if (pszTitle != nullptr)
#pragma warning(suppress: 26485)
      sData.Format(_T("%s: Length:%u,\n%s"), pszTitle, lSize, (pszString != nullptr) ? pszString : _T("<NULL>\n")); //NOLINT(clang-diagnostic-format)
    else
#pragma warning(suppress: 26485)
      sData.Format(_T("Length:%u, \n%s"), lSize, (pszString != nullptr) ? pszString : _T("<NULL>\n")); //NOLINT(clang-diagnostic-format)
    _tprintf(_T("%s"), sData.GetString());

    return true;
  }

/// @brief Method to kill any pending sessions in our session cache which have not completed authentication promptly
  void KillLongRunningSessions()
  {
    const ULONGLONG nNowTickCount{GetTickCount64()};
    std::lock_guard<std::mutex> l{m_mutexSessions};
    for (auto iter=m_Sessions.begin(); iter!=m_Sessions.end(); )
    {
      if (!iter->second->m_bPersistentSession && (nNowTickCount - iter->second->m_dwStartTickCount) > 120000)
      {
        _tprintf(_T("Removed not completed session %hs from cache\n"), iter->first.GetString());
        iter = m_Sessions.erase(iter);
      }
      else
        ++iter;
    }
  }

/// @brief Method to remove a specific named session from our session cache
  void RemoveSessionFromCache(const CStringA& sSessionID)
  {
    std::lock_guard<std::mutex> l{m_mutexSessions};
    const auto iter{m_Sessions.find(sSessionID)};
    if (iter != m_Sessions.end())
    {
      _tprintf(_T("Removed session %hs from cache\n"), sSessionID.GetString());
      m_Sessions.erase(iter);
    }
  }

/// @brief Method to log the details of accounts associated with the HTTP server and the impersonated client
  void DisplayAccountDetails(_In_ PCtxtHandle phContext)
  {
    DWORD cbUserName{0};
    GetUserName(nullptr, &cbUserName);
    std::vector<TCHAR> sUserName{cbUserName, std::allocator<TCHAR>{}};
    if (!GetUserName(sUserName.data(), &cbUserName))
      _tprintf(_T("Could not get the account under which the HTTP server is running, Error:%u\n"), GetLastError()); //NOLINT(clang-diagnostic-format)
    else
    {
      sUserName.resize(cbUserName);
      _tprintf(_T("Account under which the HTTP server is running: %s\n"), sUserName.data());
    }

    SECURITY_STATUS ss{ImpersonateSecurityContext(phContext)};
    if (ss >= 0)
    {
      cbUserName = 0;
      GetUserName(nullptr, &cbUserName);
      sUserName.resize(cbUserName);
      if (!GetUserName(sUserName.data(), &cbUserName))
        _tprintf(_T("Could not get the Impersonated client name, Error:%u\n"), GetLastError()); //NOLINT(clang-diagnostic-format)
      else
      {
        sUserName.resize(cbUserName);
        _tprintf(_T("Impersonated client name is: %s\n"), sUserName.data());
      }
      ss = RevertSecurityContext(phContext);
      if (ss < 0)
        _tprintf(_T("Failed to revert client security context, Error:%d\n"), ss); //NOLINT(clang-diagnostic-format)
    }
    else
      _tprintf(_T("Failed to impersonate client security context, Error:%d\n"), ss); //NOLINT(clang-diagnostic-format)
  }

/// @brief An example which implements Negotiate (i.e. Kerberos / NTLM) authentication
  HRESULT HandleNegotiateAuthentication(_In_ const HTTP_REQUEST& request)
  {
    //Kill any pending sessions in our session cache which have not completed authentication promptly
    KillLongRunningSessions();

    CStringA sAuthorizationHeader{request.Headers.KnownHeaders[HttpHeaderAuthorization].pRawValue, request.Headers.KnownHeaders[HttpHeaderAuthorization].RawValueLength};
    std::vector<BYTE> negotiateData;
    if (!ParseNegotiateAuthorizationHeader(sAuthorizationHeader, negotiateData))
    {
      _tprintf(_T("Returning 401 Negotiate Unauthorized response\n"));
      HTTP_RESPONSE response{};
      InitializeHTTPResponse(response, 401, "Unauthorized");
      AddKnownHeader(response, HttpHeaderWwwAuthenticate, "Negotiate");
      _tprintf(_T("Returning Www-Authenticate header of Negotiate\n"));
      ULONG nBytesSent{0};
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
    }
    else
    {
      _tprintf(_T("Authorization header: %hs\n"), sAuthorizationHeader.GetString());

      //Check the credentials handle
      if (m_NegotiateSS != SEC_E_OK)
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return m_NegotiateSS;
      }

      //Prepare the buffers
      std::vector<BYTE> outBuf{m_cbMaxNegotiateTokenSize, std::allocator<BYTE>{}};
      SecBuffer OutBuffer{m_cbMaxNegotiateTokenSize, SECBUFFER_TOKEN, outBuf.data()};
      SecBufferDesc OutBuffDesc{SECBUFFER_VERSION, 1, &OutBuffer};
#pragma warning(suppress: 26472)
      SecBuffer InBuffer{static_cast<unsigned long>(negotiateData.size()), SECBUFFER_TOKEN, negotiateData.data()};
      SecBufferDesc InBuffDesc{SECBUFFER_VERSION, 1, &InBuffer};
      AuditData(_T("<Negotiate IN "), static_cast<const BYTE*>(InBuffer.pvBuffer), InBuffer.cbBuffer);

      //Try to find the session identifier cookie in our session cache
      std::shared_ptr<CSecHandle> context;
      CStringA sCookiesHeader{request.Headers.KnownHeaders[HttpHeaderCookie].pRawValue, request.Headers.KnownHeaders[HttpHeaderCookie].RawValueLength};
      CStringA sNegotiateSessionID;
      if (ParseCookieFromCookieHeader(sCookiesHeader, "SSPISID", sNegotiateSessionID))
      {
        std::lock_guard<std::mutex> l{m_mutexSessions};
        const auto iter{m_Sessions.find(sNegotiateSessionID)};
        if (iter != m_Sessions.end())
        {
          _tprintf(_T("Located session from cookie of %hs\n"), sNegotiateSessionID.GetString());
          context = iter->second->m_Context;
        }
      }
      const bool bFirstPass{context.get() == nullptr};
      if (bFirstPass)
        context = std::make_shared<CSecHandle>();
      ULONG ContextAttributes{ASC_REQ_STREAM | ASC_REQ_DELEGATE};
      SECURITY_STATUS ss{AcceptSecurityContext(&m_NegotiateCred.m_Handle, bFirstPass ? nullptr : &context->m_Handle, &InBuffDesc, 0, SECURITY_NETWORK_DREP, &context->m_Handle, &OutBuffDesc, &ContextAttributes, nullptr)};
      if ((ss == SEC_I_COMPLETE_NEEDED) || (ss == SEC_I_COMPLETE_AND_CONTINUE))
      {
        ss = CompleteAuthToken(&context->m_Handle, &OutBuffDesc);
        if (ss != SEC_E_OK)
        {
          SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
          return ss;
        }
      }
      else if ((ss != SEC_I_CONTINUE_NEEDED) && (ss != SEC_E_OK))
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return ss;
      }

      AuditData(_T("<Negotiate OUT "), static_cast<const BYTE*>(OutBuffer.pvBuffer), OutBuffer.cbBuffer);

      //Base64 encode the data which we need to send to the client
      int nEncodedSize{ATL::Base64EncodeGetRequiredLength(OutBuffer.cbBuffer, ATL_BASE64_FLAG_NOCRLF)};
#pragma warning(suppress: 26472)
      std::vector<char> challengeData{static_cast<size_t>(nEncodedSize) + 1, std::allocator<char>{}}; //We allocate an extra byte so that we can null terminate the result
      if (!ATL::Base64Encode(static_cast<const BYTE*>(OutBuffer.pvBuffer), OutBuffer.cbBuffer, challengeData.data(), &nEncodedSize, ATL_BASE64_FLAG_NOCRLF))
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return ss;
      }
#pragma warning(suppress: 26446)
      challengeData[nEncodedSize] = '\0';

      if (ss == SEC_I_CONTINUE_NEEDED)
      {
        //Create the new cookie to allow us to complete the authentication the next time around
        const HRESULT hr{CreateSessionID(sNegotiateSessionID)};
        if (FAILED(hr))
        {
          SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
          return ss;
        }

        _tprintf(_T("Creating Negotiate session cookie of %hs\n"), sNegotiateSessionID.GetString());

        //Update our session cache with the new session
        {
          std::shared_ptr<CSession> session{std::make_shared<CSession>()};
          session->m_Context = context;
          session->m_dwStartTickCount = GetTickCount64();
          std::lock_guard<std::mutex> l{m_mutexSessions};
          m_Sessions[sNegotiateSessionID] = session;
        }

        _tprintf(_T("Returning 401 Negotiate Unauthorized challenge response\n"));
        HTTP_RESPONSE response{};
        InitializeHTTPResponse(response, 401, "Unauthorized");
        CStringA sWWWAuthenticate;
        sWWWAuthenticate.Format("Negotiate %s", challengeData.data());
        AddKnownHeader(response, HttpHeaderWwwAuthenticate, sWWWAuthenticate);
        _tprintf(_T("Returning Www-Authenticate header of %hs\n"), sWWWAuthenticate.GetString());
        CStringA sCookies;
        sCookies.Format("SSPISID=%s", sNegotiateSessionID.GetString());
        AddKnownHeader(response, HttpHeaderSetCookie, sCookies);
        _tprintf(_T("Returning Negotiate session cookie of %hs\n"), sCookies.GetString());
        ULONG nBytesSent{0};
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
      }
      else
      {
        //Remove the session from our session cache
        RemoveSessionFromCache(sNegotiateSessionID);

        DisplayAccountDetails(&context->m_Handle);
        _tprintf(_T("Returning 200 Negotiate authenticated response\n"));

        //Remove the cookie from the client
        HTTP_RESPONSE response{};
        AddKnownHeader(response, HttpHeaderSetCookie, "SSPISID=; expires=Thu, 01 Jan 1970 00:00:00 GMT");

        //An example html page which was authenticated properly
        static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This page is protected using Negotiate authentication</html>)"};
#pragma warning(suppress: 26472)
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse))));
      }
    }
  }

/// @brief An example which implements NTLM authentication
  HRESULT HandleNTLMAuthentication(_In_ const HTTP_REQUEST& request)
  {
    //Kill any pending sessions in our session cache which have not completed authentication promptly
    KillLongRunningSessions();

    CStringA sAuthorizationHeader{request.Headers.KnownHeaders[HttpHeaderAuthorization].pRawValue, request.Headers.KnownHeaders[HttpHeaderAuthorization].RawValueLength};
    std::vector<BYTE> ntlmData;
    if (!ParseNTLMAuthorizationHeader(sAuthorizationHeader, ntlmData))
    {
      _tprintf(_T("Returning 401 NTLM Unauthorized response\n"));
      HTTP_RESPONSE response{};
      InitializeHTTPResponse(response, 401, "Unauthorized");
      AddKnownHeader(response, HttpHeaderWwwAuthenticate, "NTLM");
      _tprintf(_T("Returning Www-Authenticate header of NTLM\n"));
      ULONG nBytesSent{0};
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
    }
    else
    {
      _tprintf(_T("Authorization header: %hs\n"), sAuthorizationHeader.GetString());

      //Check the credentials handle
      if (m_NTLMSS != SEC_E_OK)
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return m_NTLMSS;
      }

      //Prepare the buffers
      std::vector<BYTE> outBuf{m_cbMaxNTLMTokenSize, std::allocator<BYTE>{}};
      SecBuffer OutBuffer{m_cbMaxNTLMTokenSize, SECBUFFER_TOKEN, outBuf.data()};
      SecBufferDesc OutBuffDesc{SECBUFFER_VERSION, 1, &OutBuffer};
#pragma warning(suppress: 26472)
      SecBuffer InBuffer{static_cast<unsigned long>(ntlmData.size()), SECBUFFER_TOKEN, ntlmData.data()};
      SecBufferDesc InBuffDesc{SECBUFFER_VERSION, 1, &InBuffer};
      AuditData(_T("<NTLM IN "), static_cast<const BYTE*>(InBuffer.pvBuffer), InBuffer.cbBuffer);

      //Try to find the session identifier cookie in our session cache
      CStringA sCookiesHeader{request.Headers.KnownHeaders[HttpHeaderCookie].pRawValue, request.Headers.KnownHeaders[HttpHeaderCookie].RawValueLength};
      std::shared_ptr<CSecHandle> context;
      CStringA sNTLMSessionID;
      if (ParseCookieFromCookieHeader(sCookiesHeader, "SSPISID", sNTLMSessionID))
      {
        std::lock_guard<std::mutex> l{m_mutexSessions};
        const auto iter{m_Sessions.find(sNTLMSessionID)};
        if (iter != m_Sessions.end())
        {
          _tprintf(_T("Located session from cookie of %hs\n"), sNTLMSessionID.GetString());
          context = iter->second->m_Context;
        }
      }
      const bool bFirstPass{context.get() == nullptr};
      if (bFirstPass)
        context = std::make_shared<CSecHandle>();
      ULONG ContextAttributes{ASC_REQ_STREAM | ASC_REQ_DELEGATE};
      SECURITY_STATUS ss{AcceptSecurityContext(&m_NTLMCred.m_Handle, bFirstPass ? nullptr : &context->m_Handle, &InBuffDesc, 0, SECURITY_NETWORK_DREP, &context->m_Handle, &OutBuffDesc, &ContextAttributes, nullptr)};
      if ((ss == SEC_I_COMPLETE_NEEDED) || (ss == SEC_I_COMPLETE_AND_CONTINUE))
      {
        ss = CompleteAuthToken(&context->m_Handle, &OutBuffDesc);
        if (ss != SEC_E_OK)
        {
          SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
          return ss;
        }
      }
      else if ((ss != SEC_I_CONTINUE_NEEDED) && (ss != SEC_E_OK))
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return ss;
      }

      AuditData(_T("<NTLM OUT "), static_cast<const BYTE*>(OutBuffer.pvBuffer), OutBuffer.cbBuffer);

      //Base64 encode the data which we need to send to the client
      int nEncodedSize{ATL::Base64EncodeGetRequiredLength(OutBuffer.cbBuffer, ATL_BASE64_FLAG_NOCRLF)};
#pragma warning(suppress: 26472)
      std::vector<char> challengeData{static_cast<size_t>(nEncodedSize) + 1, std::allocator<char>{}}; //We allocate an extra byte so that we can null terminate the result
      if (!ATL::Base64Encode(static_cast<const BYTE*>(OutBuffer.pvBuffer), OutBuffer.cbBuffer, challengeData.data(), &nEncodedSize, ATL_BASE64_FLAG_NOCRLF))
      {
        SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
        return ss;
      }
#pragma warning(suppress: 26446)
      challengeData[nEncodedSize] = '\0';

      if (ss == SEC_I_CONTINUE_NEEDED)
      {
        //Create the new cookie to allow us to complete the authentication the next time around
        const HRESULT hr{CreateSessionID(sNTLMSessionID)};
        if (FAILED(hr))
        {
          SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
          return ss;
        }

        _tprintf(_T("Creating NTLM session cookie of %hs\n"), sNTLMSessionID.GetString());

        //Update our session cache with the new session
        {
          std::shared_ptr<CSession> session{std::make_shared<CSession>()};
          session->m_Context = context;
          session->m_dwStartTickCount = GetTickCount64();
          std::lock_guard<std::mutex> l{m_mutexSessions};
          m_Sessions[sNTLMSessionID] = session;
        }

        _tprintf(_T("Returning 401 NTLM Unauthorized challenge response\n"));
        HTTP_RESPONSE response{};
        InitializeHTTPResponse(response, 401, "Unauthorized");
        CStringA sWWWAuthenticate;
        sWWWAuthenticate.Format("NTLM %s", challengeData.data());
        AddKnownHeader(response, HttpHeaderWwwAuthenticate, sWWWAuthenticate);
        _tprintf(_T("Returning Www-Authenticate header of %hs\n"), sWWWAuthenticate.GetString());
        CStringA sCookies;
        sCookies.Format("SSPISID=%s", sNTLMSessionID.GetString());
        AddKnownHeader(response, HttpHeaderSetCookie, sCookies);
        _tprintf(_T("Returning NTLM session cookie of %hs\n"), sCookies.GetString());
        ULONG nBytesSent{0};
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
      }
      else
      {
        //Remove the session from our session cache
        RemoveSessionFromCache(sNTLMSessionID);

        DisplayAccountDetails(&context->m_Handle);
        _tprintf(_T("Returning 200 NTLM authenticated response\n"));

        //Remove the cookie from the client
        HTTP_RESPONSE response{};
        AddKnownHeader(response, HttpHeaderSetCookie, "SSPISID=; expires=Thu, 01 Jan 1970 00:00:00 GMT");

        //An example html page which was authenticated properly
        static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This page is protected using NTLM authentication</html>)"};
#pragma warning(suppress: 26472)
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse))));
      }
    }
  }

/// @brief Sends an example 401 Digest Unauthorized response
  HRESULT SendDigestUnauthorizedResponse(_In_ const HTTP_REQUEST& request)
  {
    //Create the new cookie to allow us to match a subsequent HTTP request to this HTTP response
    CStringA sDigestSessionID;
    const HRESULT hr{CreateSessionID(sDigestSessionID)};
    if (FAILED(hr))
    {
      SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      return hr;
    }

    _tprintf(_T("Creating Digest session cookie of %hs\n"), sDigestSessionID.GetString());

    CStringA sNonce{GenerateDigestNonce("")};

    //Update our session cache with the new session
    {
      std::shared_ptr<CSession> session{std::make_shared<CSession>()};
      session->m_sNonce = sNonce;
      session->m_dwStartTickCount = GetTickCount64();
      std::lock_guard<std::mutex> l{m_mutexSessions};
      m_Sessions[sDigestSessionID] = session;
    }

    _tprintf(_T("Returning 401 Digest Unauthorized response\n"));
    HTTP_RESPONSE response{};
    InitializeHTTPResponse(response, 401, "Unauthorized");
    CStringA sDigest;
    sDigest.Format(R"(Digest realm="http-auth@example.org", qop="auth", algorithm=MD5-sess, nonce="%s", opaque="%s")", sNonce.GetString(), GenerateRandom(16).GetString());
    AddKnownHeader(response, HttpHeaderWwwAuthenticate, sDigest);
    _tprintf(_T("Returning Www-Authenticate header of %hs\n"), sDigest.GetString());
    CStringA sCookies;
    sCookies.Format("DIGESTSID=%s", sDigestSessionID.GetString());
    AddKnownHeader(response, HttpHeaderSetCookie, sCookies);
    _tprintf(_T("Returning Digest session cookie of %hs\n"), sCookies.GetString());
    ULONG nBytesSent{0};
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
  }

/// @brief An example which implements Digest authentication
  HRESULT HandleDigestAuthentication(_In_ const HTTP_REQUEST& request)
  {
    //Kill any pending sessions in our session cache which have not completed authentication promptly
    KillLongRunningSessions();

    CStringA sAuthorizationHeader{request.Headers.KnownHeaders[HttpHeaderAuthorization].pRawValue, request.Headers.KnownHeaders[HttpHeaderAuthorization].RawValueLength};
    CDigestHeaderVariables variables;
    if (!ParseDigestAuthorizationHeader(sAuthorizationHeader, variables))
      return SendDigestUnauthorizedResponse(request);
    else
    {
      _tprintf(_T("Authorization header: %hs\n"), sAuthorizationHeader.GetString());
      _tprintf(_T("Digest authorization username: %hs\n"), variables.m_sUsername.GetString());
      _tprintf(_T("Digest authorization realm: %hs\n"), variables.m_sRealm.GetString());
      _tprintf(_T("Digest authorization nonce: %hs\n"), variables.m_sNonce.GetString());
      _tprintf(_T("Digest authorization uri: %hs\n"), variables.m_sUri.GetString());
      _tprintf(_T("Digest authorization algorithm: %hs\n"), variables.m_sAlgorithm.GetString());
      _tprintf(_T("Digest authorization response: %hs\n"), variables.m_sResponse.GetString());
      _tprintf(_T("Digest authorization qop: %hs\n"), variables.m_sQop.GetString());
      _tprintf(_T("Digest authorization nc: %hs\n"), variables.m_sNc.GetString());
      _tprintf(_T("Digest authorization cnonce: %hs\n"), variables.m_sCnonce.GetString());
      _tprintf(_T("Digest authorization opaque: %hs\n"), variables.m_sOpaque.GetString());

      //Try to find the session identifier cookie in our session cache
      CStringA sCookiesHeader{request.Headers.KnownHeaders[HttpHeaderCookie].pRawValue, request.Headers.KnownHeaders[HttpHeaderCookie].RawValueLength};
      CStringA sDigestSessionID;
      bool bFoundPreviousSession{false};
      CStringA sNonceFromPreviousSession;
      if (ParseCookieFromCookieHeader(sCookiesHeader, "DIGESTSID", sDigestSessionID))
      {
        std::lock_guard<std::mutex> l{m_mutexSessions};
        const auto iter{m_Sessions.find(sDigestSessionID)};
        if (iter != m_Sessions.end())
        {
          _tprintf(_T("Located session from cookie of %hs\n"), sDigestSessionID.GetString());
          sNonceFromPreviousSession = iter->second->m_sNonce;
          bFoundPreviousSession = true;
          //Note that we do not implement logic to avoid replay detection on the "nc" digest parameter. If you wanted to
          //do this, then you would update a nonceCount value in the CSession here, not expire the session below and
          //also implement logic to detect and fail replay attacks
        }
      }

      //Validate all the provided Digest parameters
      if (variables.m_sQop.IsEmpty() || variables.m_sNonce.IsEmpty() || (variables.m_sAlgorithm.CompareNoCase("MD5-sess") != 0) ||
          variables.m_sUsername.IsEmpty() || variables.m_sResponse.IsEmpty() || variables.m_sCnonce.IsEmpty() || variables.m_sNc.IsEmpty() ||
          (bFoundPreviousSession && (sNonceFromPreviousSession != variables.m_sNonce)))
      {
        _tprintf(_T("Returning 400 Digest Bad Request response\n"));
        HTTP_RESPONSE response{};
        InitializeHTTPResponse(response, 400, "Bad Request");
        ULONG nBytesSent = 0;
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
      }

      const DigestAlgorithm algorithm{DetermineDigestAlgorithm(variables)};
      CStringA sHA1{GenerateDigestHA1(algorithm, variables, "Mufasa", "Circle of Life")}; //Generate the H(A1) value from a username and password pair for demonstration purposes. Another approach would be to store just the H(A1) values directly on the server and not directly need the username and password values available at the server!
      CStringA sHA2{GenerateDigestHA2(algorithm, variables, "GET", nullptr, 0)};
      CStringA sCorrectResponse{GenerateDigestResponse(algorithm, variables, sHA1, sHA2)};
      _tprintf(_T("Correct Digest authorization response: %hs\n"), sCorrectResponse.GetString());
      if (sCorrectResponse != variables.m_sResponse)
        return SendDigestUnauthorizedResponse(request);
      else
      {
        //Remove the session from our session cache
        RemoveSessionFromCache(sDigestSessionID);

        //Remove the cookie from the client
        HTTP_RESPONSE response{};
        AddKnownHeader(response, HttpHeaderSetCookie, "DIGESTSID=; expires=Thu, 01 Jan 1970 00:00:00 GMT");

        _tprintf(_T("Returning 200 Digest authentication response\n"));
        static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This page is protected using Digest authentication</html>)"};
#pragma warning(suppress: 26472)
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(request.RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse))));
      }
    }
  }

/// @brief An example which implements basic authentication
  HRESULT HandleBasicAuthentication(_In_ const HTTP_REQUEST& request)
  {
    CStringA sAuthorizationHeader{request.Headers.KnownHeaders[HttpHeaderAuthorization].pRawValue, request.Headers.KnownHeaders[HttpHeaderAuthorization].RawValueLength};
    CStringA sUsername;
    CStringA sPassword;
    _tprintf(_T("Authorization header: %hs\n"), sAuthorizationHeader.GetString());
    if (!ParseBasicAuthorizationHeader(sAuthorizationHeader, sUsername, sPassword) || (sUsername != "admin") || (sPassword != "password"))
    {
      _tprintf(_T("Returning 401 Basic Unauthorized authentication response\n"));
      HTTP_RESPONSE response{};
      InitializeHTTPResponse(response, 401, "Unauthorized");
      AddKnownHeader(response, HttpHeaderWwwAuthenticate, "Basic realm=Basic authentication test");
      _tprintf(_T("Returning Www-Authenticate header of Basic realm=Basic authentication test\n"));
      ULONG nBytesSent{0};
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(request.RequestId, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
    }
    else
    {
      _tprintf(_T("Returning 200 Basic authentication response\n"));
      static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This page is protected using Basic authentication</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(request.RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse))));
    }
  }

/// @brief the example which implements the cookie authentication demo page
  HRESULT HandleCookieAuthentication(_In_ const HTTP_REQUEST& request) noexcept
  {
    static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head>)"
                                                   R"(<p><h1>Login</h1><form action="./LoginCookie" method="post">)"
                                                   R"(<label for="username">Username: </label><input type="text" id="username" name="username" value=""><br>)"
                                                   R"(<label for="password">Password: </label><input type="password" id="password" name="password" value=""><br>)"
                                                   R"(<input type="submit" value="Submit"></form>)"
                                                   R"(<p><h1>Logout</h1><form action="./LogoutCookie" method="post">)"
                                                   R"(<input type="submit" value="Submit"></form>)"
                                                   R"(<p><h1>Create Account</h1><form action="./CreateAccount" method="post">)"
                                                   R"(<label for="username">Username: </label><input type="text" id="username" name="username" value=""><br>)"
                                                   R"(<label for="password">Password: </label><input type="password" id="password" name="password" value=""><br>)"
                                                   R"(<input type="submit" value="Submit"></form>)"
                                                   R"(</html>)"};
#pragma warning(suppress: 26472)
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(request.RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse))));
  }

/// @brief the example which handles the cookie authentication home page
  HRESULT HandleCookieHomepage(_In_ const HTTP_REQUEST& request)
  {
    //Validate the session id from the cookie header
    CStringA sCookiesHeader{request.Headers.KnownHeaders[HttpHeaderCookie].pRawValue, request.Headers.KnownHeaders[HttpHeaderCookie].RawValueLength};
    bool bValidUser{false};
    CStringA sCookieSessionID;
    if (ParseCookieFromCookieHeader(sCookiesHeader, "SID", sCookieSessionID))
    {
      std::lock_guard<std::mutex> l{m_mutexSessions};
      const auto iter{m_Sessions.find(sCookieSessionID)};
      if (iter != m_Sessions.end())
      {
        _tprintf(_T("Located session from cookie of \"%hs\" on cookie homepage\n"), sCookieSessionID.GetString());
        bValidUser = true;
      }
    }
    if (!bValidUser)
    {
      _tprintf(_T("Failed to locate session from cookie header of \"%hs\" on cookie homepage\n"), sCookiesHeader.GetString());
      HTTP_RESPONSE response{};
      CStringA sLocation;
      sLocation.Format("%s?cookie", CStringA(request.CookedUrl.pFullUrl).GetString());
      AddKnownHeader(response, HttpHeaderLocation, sLocation);
      return SendResponse(response, request.RequestId, 302, "Found", nullptr, nullptr, 0);
    }

    static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This page is the home page for Cookie authentication</html>)"};
#pragma warning(suppress: 26472)
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(request.RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse))));
  }

/// @brief the example which handles create cookie authentication account requests
  HRESULT HandleCreateAccount(_In_ const HTTP_REQUEST& request)
  {
    //parse the form variables
    std::map<CString, CString, CTCharCompare> keyValues;
    HRESULT hr{ParseFormVariables(request, keyValues)};
    if (FAILED(hr))
    {
      SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      return hr;
    }

    //Validate the form variables
    const auto username{keyValues.find(_T("username"))};
    if ((username == keyValues.end()) || (username->second.GetLength() == 0))
    {
      _tprintf(_T("Username was not valid\n"));

      HTTP_RESPONSE response{};
      static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>The supplied username was not valid. Please try again.</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
    }
    const CString& sUsername{username->second};
    const auto password{keyValues.find(_T("password"))};
    if ((password == keyValues.end()) || (password->second.GetLength() == 0))
    {
      _tprintf(_T("Password was not valid\n"));

      HTTP_RESPONSE response{};
      static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This supplied password was not valid. Please try again.</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
    }

    //Check the username does not already exist in our session cache
    bool bExistingUser{false};
    {
      std::lock_guard<std::mutex> l{m_mutexSessions};
      auto iter{std::find_if(m_Sessions.cbegin(), m_Sessions.cend(), [&sUsername](const auto& element) noexcept
        {
          return element.second->m_sUsername == sUsername;
        })};
      if (iter != m_Sessions.cend())
        bExistingUser = true;
    }
    if (bExistingUser)
    {
      _tprintf(_T("Username is already in use\n"));

      HTTP_RESPONSE response{};
      static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This username is already in use. Please try again.</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
    }

    //Create the new cookie to associate with the new session
    CStringA sCookieSessionID;
    hr = CreateSessionID(sCookieSessionID);
    if (FAILED(hr))
    {
      SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      return hr;
    }

    _tprintf(_T("Creating new session of %hs with username: \"%s\"\n"), sCookieSessionID.GetString(), sUsername.GetString());

    //Update our session cache with the new session. Note a real world implementation would persist the account details
    //in some form of non-volatile storage such as a database at this point so that accounts would be persistent across
    //restarts of the web server.
    {
      std::shared_ptr<CSession> session{std::make_shared<CSession>()};
      session->m_sUsername = sUsername;
      session->m_sNonce = GenerateRandom(16); //Use CSession::m_sNonce as the location for salting the password hash
      CString sDataToHash{password->second};
      sDataToHash += _T(":");
      sDataToHash += session->m_sNonce;
#pragma warning(suppress: 26490)
      session->m_sPasswordHash = Hash(BCRYPT_SHA256_ALGORITHM, reinterpret_cast<const BYTE*>(sDataToHash.GetString()), sDataToHash.GetLength() * sizeof(TCHAR));
      session->m_dwStartTickCount = GetTickCount64();
      session->m_bPersistentSession = true;
      std::lock_guard<std::mutex> l{m_mutexSessions};
      m_Sessions[sCookieSessionID] = session;
    }

    HTTP_RESPONSE response{};
    static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This new account has been created. Click <a href=".?cookie">here</a> to login.</html>)"};
#pragma warning(suppress: 26472)
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
  }

/// @brief the example which handles cookie authentication logout requests
  HRESULT HandleLogoutCookie(_In_ const HTTP_REQUEST& request) noexcept
  {
    _tprintf(_T("Logging out user\n"));

    HTTP_RESPONSE response{};
    //Remove the cookie from the client
    AddKnownHeader(response, HttpHeaderSetCookie, "SID=; expires=Thu, 01 Jan 1970 00:00:00 GMT");
    static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>You have been logged out. Click <a href=".?cookie">here</a> to login again.</html>)"};
#pragma warning(suppress: 26472)
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
  }

/// @brief the example which handles cookie authentication login requests
  HRESULT HandleLoginCookie(_In_ const HTTP_REQUEST& request)
  {
    _tprintf(_T("Doing cookie login authentication\n"));

    //parse the form variables
    std::map<CString, CString, CTCharCompare> keyValues;
    const HRESULT hr{ParseFormVariables(request, keyValues)};
    if (FAILED(hr))
    {
      SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      return hr;
    }

    //Validate the form variables
    const auto username{keyValues.find(_T("username"))};
    if ((username == keyValues.end()) || (username->second.GetLength() == 0))
    {
      _tprintf(_T("Username was not valid\n"));

      HTTP_RESPONSE response{};
      static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>The supplied username was not valid. Please try again.</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
    }
    const CString& sUsername{username->second};
    const auto password{keyValues.find(_T("password"))};
    if ((password == keyValues.end()) || (password->second.GetLength() == 0))
    {
      _tprintf(_T("Password was not valid\n"));

      HTTP_RESPONSE response{};
      static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>This supplied password was not valid. Please try again.</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
    }

    //Check the supplied credentials against our session cache
    CStringA sCookieForUser;
    {
      std::lock_guard<std::mutex> l{m_mutexSessions};
      auto iter{std::find_if(m_Sessions.cbegin(), m_Sessions.cend(), [&sUsername, &password](const auto& element)
        {
          if (element.second->m_sUsername != sUsername)
            return false;
          CString sDataToHash{password->second};
          sDataToHash += _T(":");
          sDataToHash += element.second->m_sNonce;
#pragma warning(suppress: 26490)
          CStringA sPasswordHash{Hash(BCRYPT_SHA256_ALGORITHM, reinterpret_cast<const BYTE*>(sDataToHash.GetString()), sDataToHash.GetLength() * sizeof(TCHAR))};
          return element.second->m_sPasswordHash == sPasswordHash;
        })};
      if (iter != m_Sessions.cend())
        sCookieForUser = iter->first;
    }
    if (sCookieForUser.IsEmpty())
    {
      _tprintf(_T("Login attempt was unsuccessful\n"));

      HTTP_RESPONSE response{};
      static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>An error occurred logging you in. Please try again.</html>)"};
#pragma warning(suppress: 26472)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
    }

    _tprintf(_T("Login attempt was successful\n"));

    //Tell the user they are now logged in, along with a cookie for their session. Also provide a link to the protected home page in the response.
    HTTP_RESPONSE response{};
    CStringA sCookies;
    sCookies.Format("SID=%s", sCookieForUser.GetString());
    AddKnownHeader(response, HttpHeaderSetCookie, sCookies);
    static constexpr const char* pszResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><p>You are now logged in. Click <a href=".?cookie_homepage">here</a> to access the home page.</html>)"};
#pragma warning(suppress: 26472)
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, SendResponse(response, request.RequestId, 200, "OK", "text/html", pszResponse, static_cast<ULONG>(strlen(pszResponse))));
  }

#pragma warning(suppress: 26429)
/// @brief The main callback method which handles individual HTTP requests
  void HandleRequest(_In_ PHTTP_REQUEST pRequest) override
  {
    //Validate our parameters
    ATLASSERT(pRequest != nullptr); //NOLINT(clang-analyzer-core.CallAndMessage)

    switch (pRequest->Verb) //NOLINT(clang-analyzer-core.CallAndMessage)
    {
      case HttpVerbGET:
      {
        _tprintf(_T("%04X, Got GET request for %ws\n"), GetCurrentThreadId(), pRequest->CookedUrl.pFullUrl); //NOLINT(clang-diagnostic-format)
        bool bDefaultPage{false};
        if (pRequest->CookedUrl.pQueryString != nullptr)
        {
          if (wcscmp(pRequest->CookedUrl.pQueryString, L"?dir_listing") == 0)
            HandleDirectoryListing(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?file") == 0)
            HandleFileRequest(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?basic") == 0)
            HandleBasicAuthentication(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?ntlm") == 0)
            HandleNTLMAuthentication(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?negotiate") == 0)
            HandleNegotiateAuthentication(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?digest") == 0)
            HandleDigestAuthentication(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?cookie") == 0)
            HandleCookieAuthentication(*pRequest);
          else if (wcscmp(pRequest->CookedUrl.pQueryString, L"?cookie_homepage") == 0)
            HandleCookieHomepage(*pRequest);
          else
            bDefaultPage = true;
        }
        else
          bDefaultPage = true;
        if (bDefaultPage)
        {
          //An example html page which allows us to test POST support
          static constexpr const char* pszSampleResponse{R"(<html><head><meta charset="UTF-8"><title>Demo for HTTPServerWrappers</title></head><body>)"
                                                         R"(<p>Your first web server written using HTTPServerWrappers</p>)"
                                                         R"(<p>1. Post example<form action="./postecho" method="post"><label for="fname">Data:</label><br><input type="text" id="data" name="data" value="test"><input type="submit" value="Submit"></form></p>)"
                                                         R"(<p><a href=".?basic">2. Basic authentication example</a></p>)"
                                                         R"(<p><a href=".?ntlm">3. NTLM authentication example</a></p>)"
                                                         R"(<p><a href=".?negotiate">4. Negotiate authentication example</a></p>)"
                                                         R"(<p><a href=".?digest">5. Digest authentication example</a></p>)"
                                                         R"(<p><a href=".?cookie">5. Cookie authentication example</a></p>)"
                                                         R"(<p><a href=".?dir_listing">6. Directory listing and Get file example</a></p>)"
                                                         R"(</body></html>)"};
#pragma warning(suppress: 26472)
          SendResponse(pRequest->RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse)));
        }
        break;
      }
      case HttpVerbPOST:
      {
        _tprintf(_T("%04X, Got POST request for %ws\n"), GetCurrentThreadId(), pRequest->CookedUrl.pFullUrl); //NOLINT(clang-diagnostic-format)
        CStringW sAbsPath{pRequest->CookedUrl.pAbsPath};
        if (sAbsPath.Find(L"postecho") != -1)
          HandleEchoPost(*pRequest);
        else if (sAbsPath.Find(L"CreateAccount") != -1)
          HandleCreateAccount(*pRequest);
        else if (sAbsPath.Find(L"LoginCookie") != -1)
          HandleLoginCookie(*pRequest);
        else if (sAbsPath.Find(L"LogoutCookie") != -1)
          HandleLogoutCookie(*pRequest);
        else
          SendResponse(pRequest->RequestId, 404, "Not found", nullptr, nullptr, 0);
        break;
      }
      default:
      {
        _tprintf(_T("%04X, Got an unknown request for %ws\n"), GetCurrentThreadId(), pRequest->CookedUrl.pFullUrl); //NOLINT(clang-diagnostic-format)
        //An example which demonstrates a specific HTTP status code
        SendResponse(pRequest->RequestId, 503, "Not Implemented", nullptr, nullptr, 0);
        break;
      }
    }
  }

//Member variables
  CCredHandle m_NTLMCred;
  SECURITY_STATUS m_NTLMSS;
  unsigned long m_cbMaxNTLMTokenSize;
  CCredHandle m_NegotiateCred;
  SECURITY_STATUS m_NegotiateSS;
  unsigned long m_cbMaxNegotiateTokenSize;
  std::map<CStringA, std::shared_ptr<CSession>, CCharCompare> m_Sessions;
  std::mutex m_mutexSessions;
};


#pragma warning(suppress: 6262 26429 26440 26461)
int _tmain(int argc, TCHAR* argv[]) //NOLINT(modernize-avoid-c-arrays)
{
  //Parse the command line
  if (argc < 2)
  {
    _tprintf(_T("Usage: testHTTPServerWrappers <URL1> [URL2]..."));
    return 0;
  }

  //Initialize the HTTP Server APIs using the RAII class
  HTTPServer::CAutoInit init;
  const ULONG nRetCode{init.Init()};
  if (nRetCode != NO_ERROR)
  {
    _tprintf(_T("Failed to initialize HTTP Server API, Error:%u\n"), nRetCode); //NOLINT(clang-diagnostic-format)
    return nRetCode;
  }

//#define UNIT_TEST_CPPCONCURRENCY_CTHREADPOOL

#ifdef UNIT_TEST_CPPCONCURRENCY_CTHREADPOOL
  //Unit test the thread pool framework. Some of the tests are based on the unit tests
  //at https://github.com/f-squirrel/thread_pool/blob/master/tests/ThreadPoolTest.cpp
  //which is also based on the thread_pool class from "C++ Concurrency In Action"
  _tprintf(_T("***Starting unit tests of the CppConcurrency::CThreadPool framework***!\n"));

{
  _tprintf(_T("Test WaitForInitInstanceThreads (thread count == 1)\n"));

  CThreadPoolWithStartupFailures pool{1};
  ATLASSERT(!pool.started());
  pool.start();
  ATLASSERT(pool.started());
  const bool bSuccess{pool.WaitForInitInstanceThreads()};
  _tprintf(_T(" InitInstance value from the thread pool was %d\n"), static_cast<int>(bSuccess));
}

{
  _tprintf(_T("Test Test InitInstanceThread (thread count == 6)\n"));

  CThreadPoolWithCustomInitInstanceThread pool{6};
  ATLASSERT(!pool.started());
  pool.start();
  ATLASSERT(pool.started());
  const bool bSuccess{pool.WaitForInitInstanceThreads()};
  ATLASSERT(bSuccess);
  UNREFERENCED_PARAMETER(bSuccess);
  ATLASSERT(pool._ThreadsWhichRanInitInstanceThread == 6);
}

{
  _tprintf(_T("Test WaitForInitInstanceThreads (thread count == 5)\n"));

  CThreadPoolWithStartupFailures pool{5};
  pool.start();
  const bool bSuccess{pool.WaitForInitInstanceThreads()};
  _tprintf(_T(" InitInstance value from the thread pool was %d\n"), static_cast<int>(bSuccess));
}

{
  _tprintf(_T("Test CheckDefaultNumberOfThreads\n"));

  CppConcurrency::CThreadPool pool;
  pool.start();
  ATLASSERT(pool.size() == std::thread::hardware_concurrency());
}

{
  _tprintf(_T("Test CheckNumberOfThreadsIfHardwareConcurrency0\n"));

  CppConcurrency::CThreadPool pool{0};
  pool.start();
  ATLASSERT(pool.size() == std::thread::hardware_concurrency());
}

{
  _tprintf(_T("Test running more tasks than threads\n"));

  constexpr size_t THREAD_COUNT{2u};
  constexpr size_t TASK_COUNT{20u};

  std::mutex mutex;
  size_t result{0u};
  std::set<std::thread::id> thread_ids;

  CppConcurrency::CThreadPool pool{THREAD_COUNT};
  pool.start();
  std::vector<std::future<void>> futures;
  for (size_t i=0; i<TASK_COUNT; ++i)
  {
    futures.emplace_back(pool.submit([&] {
      std::this_thread::sleep_for(std::chrono::milliseconds(1));
      std::lock_guard<std::mutex> l{mutex};
      ++result;
      thread_ids.insert(std::this_thread::get_id());
      }));
  }
  for (auto& f : futures)
    f.wait();
  ATLASSERT(pool.queueSize() == 0);
  ATLASSERT(result == TASK_COUNT);
  ATLASSERT(thread_ids.size() == THREAD_COUNT);
}

{
  _tprintf(_T("Test VariousTypesOfTasks\n"));

  CppConcurrency::CThreadPool pool{2u};
  pool.start();
  auto fi{pool.submit([] { return 42; })};
  auto fs{pool.submit([] { return std::string{"42"}; })};
  ATLASSERT(fi.get() == 42);
  ATLASSERT(fs.get() == std::string{ "42" });
}

{
  _tprintf(_T("Test Lambdas submissions\n"));

  constexpr size_t TASK_COUNT{4u};
  std::vector<std::future<size_t>> v;
  CppConcurrency::CThreadPool pool{4u};
  pool.start();
  for (size_t i=0; i<TASK_COUNT; ++i)
  {
    v.emplace_back(pool.submit([task_num = i] {
      std::this_thread::sleep_for(std::chrono::milliseconds(1));
      return task_num;
      }));
  }
  for (size_t i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    ATLASSERT(i == v[i].get());
  }
}

{
  _tprintf(_T("Test std::refs and std::crefs submissions\n"));
  int nResult{0};
#pragma warning(suppress: 26814)
  const int nAddition{12};
  CppConcurrency::CThreadPool pool;
  pool.start();
  pool.submit(accumulate_with_refs, 42, std::cref(nAddition), std::ref(nResult)).get();
  ATLASSERT(nResult == 54);
}

{
  _tprintf(_T("Test Exception handling using a lambda\n"));

  CppConcurrency::CThreadPool pool{1u};
  pool.start();
  auto f{pool.submit([] { throw std::runtime_error{ "Error" }; })};
  try
  {
    f.get();
  }
  catch (std::runtime_error&)
  {
  }
  catch (...)
  {
    ATLASSERT(FALSE);
  }
}

{
  _tprintf(_T("Test Exception handling using a function\n"));

  CppConcurrency::CThreadPool pool{1u};
  pool.start();
  auto f{pool.submit(throwRuntimeError)};

  try
  {
    f.get();
  }
  catch (std::runtime_error&)
  {
  }
  catch (...)
  {
    ATLASSERT(FALSE);
  }
}

{
  _tprintf(_T("Test size\n"));

  CppConcurrency::CThreadPool pool{4u};
  pool.start();
  ATLASSERT(pool.size() == 4u);
}

{
  _tprintf(_T("Test empty queue\n"));

  CppConcurrency::CThreadPool pool{4u};
  pool.start();
  std::this_thread::sleep_for(std::chrono::seconds(1));
}

{
  _tprintf(_T("Test initially paused thread pool\n"));

  CppConcurrency::CThreadPool pool{4u, true, true};
  pool.start();
  ATLASSERT(pool.paused());
}

{
  _tprintf(_T("Test pause method\n"));

  CppConcurrency::CThreadPool pool{4u, true, true};
  pool.start();
  pool.pause(false);
  ATLASSERT(!pool.paused());
  pool.pause(true);
  ATLASSERT(pool.paused());
}

{
  _tprintf(_T("Test pause method take 2\n"));

  constexpr size_t THREAD_COUNT{4u};
  constexpr int TASK_COUNT{80};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, false, false};
  pool.start();

  _tprintf(_T(" Submitting tasks prior to pause\n"));
  for (size_t i=0; i<TASK_COUNT; ++i)
    pool.submit(test_function, i * 10);
  std::this_thread::sleep_for(std::chrono::milliseconds(20));
  _tprintf(_T(" Pausing\n"));
  pool.pause(true);
  _tprintf(_T(" Pool now paused\n"));
  for (int i=0; i<40; i++)
  {
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(20));
  }
  _tprintf(_T(" Unpausing\n"));
  pool.pause(false);
  for (int i=0; i<40; i++)
  {
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(200));
  }
}

{
  _tprintf(_T("Test clear method\n"));

  CppConcurrency::CThreadPool pool{1u};
  pool.start();
  for (int i=0; i<1000; i++)
    pool.submit([] { return 42; } );
#pragma warning(suppress: 26472)
  _tprintf(_T(" main thread pool queue size:%d\n"), static_cast<int>(pool.queueSize()));
  pool.clear();
  const auto size{pool.queueSize()};
  ATLASSERT(size == 0);
#pragma warning(suppress: 26472)
  _tprintf(_T(" main thread pool queue size:%d\n"), static_cast<int>(size));
}

{
  _tprintf(_T("Test function with arguments\n"));

  CppConcurrency::CThreadPool pool{4u};
  pool.start();
  auto f{pool.submit(sum, 2, 2)};
  ATLASSERT(f.get() == 4);
}

{
  _tprintf(_T("Test get_thread method\n"));

  constexpr unsigned THREAD_COUNT{4u};
  CppConcurrency::CThreadPool pool{THREAD_COUNT};
  pool.start();
  for (unsigned i=0; i<THREAD_COUNT; ++i)
  {
    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << pool.get_thread(i).get_id();
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Thread index:%u and processed on thread id:0x%04x\n"), i, nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test resize method\n"));

  _tprintf(_T(" thread pool initially created with 4 thread\n"));
  CppConcurrency::CThreadPool pool{4u};
  pool.start();
  for (unsigned i=0; i<4u; ++i)
  {
    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << pool.get_thread(i).get_id();
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Thread index:%u and thread id:0x%04x\n"), i, nID); //NOLINT(clang-diagnostic-format)
  }
  std::vector<std::future<std::thread::id>> futures;
  constexpr unsigned TASK_COUNT{40};
  for (size_t i=0; i<TASK_COUNT; ++i)
    futures.emplace_back(pool.submit(test_function, i * 10));
  std::this_thread::sleep_for(std::chrono::milliseconds(10));
  _tprintf(_T(" resizing thread pool to 2 threads\n"));
  pool.resize(2);
  ATLASSERT(pool.size() == 2);
  for (unsigned i=0; i<2u; ++i)
  {
    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << pool.get_thread(i).get_id();
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Thread index:%u and thread id:0x%04x\n"), i, nID); //NOLINT(clang-diagnostic-format)
  }
  for (size_t i=0; i<TASK_COUNT; ++i)
    futures.emplace_back(pool.submit(test_function, i * 10));
  std::this_thread::sleep_for(std::chrono::milliseconds(10));
  _tprintf(_T(" resizing thread pool to 5 threads\n"));
  pool.resize(5);
  ATLASSERT(pool.size() == 5);
  for (unsigned i=0; i<5; ++i)
  {
    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << pool.get_thread(i).get_id();
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Thread index:%u and thread id:0x%04x\n"), i, nID); //NOLINT(clang-diagnostic-format)
  }
  for (size_t i=0; i<TASK_COUNT; ++i)
    futures.emplace_back(pool.submit(test_function, i * 10));
  for (unsigned i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was processed on thread id:0x%04x\n"), nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test task submission to a specific thread without work stealing\n"));

  constexpr size_t THREAD_COUNT{4u};
  constexpr unsigned TASK_COUNT{40};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, false};
  pool.start();
  std::vector<std::future<std::thread::id>> futures;
  for (size_t i=0; i<TASK_COUNT; ++i)
    futures.emplace_back(pool.submit(i % THREAD_COUNT, test_function, i * 10));
  for (unsigned i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was submitted on thread index:%u and processed on thread id:0x%04x\n"), static_cast<unsigned>(i % THREAD_COUNT), nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test task submission to a specific thread with task stealing\n"));

  constexpr size_t THREAD_COUNT{4u};
  constexpr unsigned TASK_COUNT{40};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, true};
  pool.start();
  std::vector<std::future<std::thread::id>> futures;

  constexpr int TasksPerThread{TASK_COUNT / THREAD_COUNT};
  for (size_t i=0; i<THREAD_COUNT; ++i)
  {
    for (int j=0; j<TasksPerThread; ++j)
#pragma warning(suppress: 26472)
      futures.emplace_back(pool.submit(static_cast<unsigned>(i), test_function, i * 100));
  }
  for (unsigned i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was submitted on thread index:%u and processed on thread id:0x%04x\n"), static_cast<unsigned>(i % THREAD_COUNT), nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test task submission with task stealing and resubmission from thread pool worker threads + queueSize  + threadQueueSize + threadsIdle methods\n"));

  constexpr size_t THREAD_COUNT{4u};
  constexpr int TASK_COUNT{80};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, true};
  pool.start();
  std::vector<std::future<std::thread::id>> futures;

  constexpr int TasksPerThread{TASK_COUNT / THREAD_COUNT};
  for (size_t i=0; i<THREAD_COUNT; ++i)
  {
    for (int j=0; j<TasksPerThread; ++j)
      futures.emplace_back(pool.submit(test_submitfromworkerthread, &pool, i * 100));
  }
  for (int i=0; i<80; i++)
  {
    using namespace std::chrono_literals;
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(10));
  }
  for (unsigned i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was submitted on thread index:%u and processed on thread id:0x%04x\n"), static_cast<unsigned>(i % THREAD_COUNT), nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test custom WorkerThread method\n"));

  CThreadPoolWithCustomeWorkerThread pool{1};
  pool.start();
  pool.WaitForInitInstanceThreads();
}

{
  _tprintf(_T("Test threads are reused\n"));

  constexpr size_t THREAD_COUNT{4u};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, false};
  pool.start();
  std::vector<std::future<std::thread::id>> futures;
  std::set<std::thread::id> thread_ids;
  for (size_t i=0; i<THREAD_COUNT; ++i)
    futures.emplace_back(pool.submit(test_function, i));
  for (size_t i=0; i<THREAD_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was processed on thread id:0x%04x\n"), nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test blocked_submit 1\n"));

  constexpr size_t THREAD_COUNT{4u};
  constexpr int TASK_COUNT{80};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, false, false};
  pool.start();

  _tprintf(_T(" Submitting tasks without blocking\n"));
  for (size_t i=0; i<TASK_COUNT; ++i)
    pool.submit(test_function, i * 10);
  for (int i=0; i<10; i++)
  {
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(20));
  }
  _tprintf(_T(" Submitting tasks with blocking, 60 is max queue size value used\n"));
  for (size_t i=0; i<10; ++i)
  {
    pool.blocked_submit(60, test_function, i * 10);
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(40));
  }
  for (size_t i=0; i<60; ++i)
  {
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(10));
  }
}

{
  _tprintf(_T("Test blocked_submit 2\n"));

  constexpr size_t THREAD_COUNT{4u};
  constexpr int TASK_COUNT{80};
  CppConcurrency::CThreadPool pool{THREAD_COUNT, true, false};
  pool.start();

  _tprintf(_T(" Submitting tasks without blocking\n"));
  for (size_t i=0; i<TASK_COUNT; ++i)
    pool.submit(i % THREAD_COUNT, test_function, i * 10);
  for (int i=0; i<10; i++)
  {
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(20));
  }
  _tprintf(_T(" Submitting tasks with blocking, 15 is max queue size value used\n"));
  for (size_t i=0; i<10; ++i)
  {
    pool.blocked_submit(15, i % THREAD_COUNT, test_function, i * 10);
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(10));
  }
  for (size_t i=0; i<60; ++i)
  {
#pragma warning(suppress: 26472)
    _tprintf(_T(" main thread pool queue size:%d, threads busy:%d, thread queue sizes:"), static_cast<int>(pool.queueSize()), static_cast<int>(pool.threadsBusy()));
    for (unsigned j=0; j<THREAD_COUNT; j++)
    {
#pragma warning(suppress: 26472)
      _tprintf(_T("%d"), static_cast<int>(pool.threadQueueSize(j)));
      if (j < (THREAD_COUNT - 1))
        _tprintf(_T(" "));
    }
    _tprintf(_T("\n"));
    std::this_thread::sleep_for(std::chrono::milliseconds(60));
  }
}

{
  _tprintf(_T("Test pumpWindowsMessage functionality 1\n"));
  CppConcurrency::CThreadPool pool{4u, true, false, true};
  pool.start();
  for (unsigned i=0; i<4u; ++i)
  {
    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << pool.get_thread(i).get_id();
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Thread index:%u and thread id:0x%04x\n"), i, nID); //NOLINT(clang-diagnostic-format)
  }
  std::vector<std::future<std::thread::id>> futures;
  constexpr unsigned TASK_COUNT{40};
  for (size_t i=0; i<TASK_COUNT; ++i)
    futures.emplace_back(pool.submit(test_function, i * 10));
  for (unsigned i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was processed on thread id:0x%04x\n"), nID); //NOLINT(clang-diagnostic-format)
  }
}

{
  _tprintf(_T("Test pumpWindowsMessage functionality with pause\n"));
  CppConcurrency::CThreadPool pool{4u, true, false, true};
  pool.start();
  for (unsigned i=0; i<4u; ++i)
  {
    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << pool.get_thread(i).get_id();
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Thread index:%u and thread id:0x%04x\n"), i, nID); //NOLINT(clang-diagnostic-format)
  }
  std::vector<std::future<std::thread::id>> futures;
  constexpr unsigned TASK_COUNT{40};
  for (size_t i=0; i<20; ++i)
    futures.emplace_back(pool.submit(test_function, i * 10));
  pool.pause(true);
  for (size_t i=0; i<20; ++i)
    futures.emplace_back(pool.submit(test_function, i * 10));
  pool.pause(false);
  for (unsigned i=0; i<TASK_COUNT; ++i)
  {
#pragma warning(suppress: 26446)
    auto r{futures[i].get()};

    //Convert the std::thread:id into a Win32 thread identifier
    std::stringstream ss;
    ss << r;
    const int nID{std::stoi(ss.str())};
#pragma warning(suppress: 26472)
    _tprintf(_T(" Task was processed on thread id:0x%04x\n"), nID); //NOLINT(clang-diagnostic-format)
  }
}

  _tprintf(_T("***Completed unit testing CppConcurrency::CThreadPool framework***\n"));
#endif //#ifdef UNIT_TEST_CPPCONCURRENCY_CTHREADPOOL

  //Build up the array of URLs to listen for
  std::vector<std::wstring> URLs;
  for (int i=1; i<argc; i++)
  {
#pragma warning(suppress: 26481)
    _tprintf(_T("Listening on URL: %s\n"), argv[i]);
#pragma warning(suppress: 26481)
    URLs.emplace_back(ATL::CT2W{argv[i]});
  }

  //Start up the HTTP server
  CSampleHTTPServer server;
  const HRESULT hr{server.Start(URLs)};
  if (FAILED(hr))
  {
    _tprintf(_T("Failed to start web server, Error:%08X\n"), hr); //NOLINT(clang-diagnostic-format)
    return hr;
  }

  //Wait around to exit
  _tprintf(_T("Press any key to shutdown!\n"));
  while (!_kbhit())
    std::this_thread::sleep_for(std::chrono::milliseconds(1000));

  return 0;
}
