/*
Module : HTTPServerWrappers.h
Purpose: Wrapper classes for the Windows HTTP Server v2 APIs
History: PJN / 08-11-2020 1. Initial Public release.
         PJN / 21-12-2020 1. Added a new override of the CServer::SendResponse method which allows an already
                          initialized HTTP_RESPONSE structure to be provided as a parameter.
                          2. Provided a new CServer::EqualsSplit method to aid parsing HTTP header values.
                          3. Provided a new CServer::IsInQuotedString method to aid parsing HTTP header values.
                          4. Provided a new CServer::HeaderTokenize method to aid parsing HTTP header values.
                          5. Provided a new CServer::UnquoteHeaderValue method to aid parsing HTTP header values.
                          6. Provided a new CServer::ParseCookieFromCookieHeader method to aid parsing HTTP cookies.
                          7. Provided a new CServer::ParseBasicAuthorizationHeader method to aid parsing Basic 
                          authentication headers.
                          8. Provided a new CServer::ParseAuthorizationHeader method to aid parsing NTLM and 
                          Negotiate authentication headers.
                          9. Provided a new CServer::ParseNegotiateAuthorizationHeader method to aid parsing 
                          Negotiate authentication headers.
                          10. Provided a new CServer::ParseNTLMAuthorizationHeader method to aid parsing 
                          NTLM authentication headers.
                          11. Provided a new CServer::DetermineDigestAlgorithm to aid parsing Digest authentication 
                          headers.
                          12. Provided a new CServer::GenerateRandom method to aid generating random data for Digest
                          authentication headers.
                          13. Provided a new CServer::Hash method to aid generating cryptographic hashes data for 
                          Digest authentication headers.
                          14. Provided new CServer::MD5 methods to aid generating MD5 hashes data for Digest 
                          authentication headers.
                          15. Provided a new CServer::GenerateDigestNonce method to aid generating cryptographic 
                          nonces for Digest authentication headers.
                          16. Provided a new CServer::GenerateDigestHA1 method to aid validating Digest 
                          authentication headers.
                          17. Provided a new CServer::GenerateDigestHA2 method to aid validating Digest 
                          authentication headers.
                          18. Provided a new CServer::GenerateDigestResponse method to aid validating Digest
                          authentication headers.
                          19. Provided a new CServer::ParseDigestAuthorizationHeader method to aid parsing Digest
                          authentication headers.
                          20. Provided a new CCredHandle RAII class in CCredHandle.h to encapsulate an SDK 
                          CredHandle handle value.
                          21. Provided a new CSecHandle RAII class in CSecHandle.h to encapsulate an SDK 
                          SecHandle handle value.
                          22. Updated the existing basic authentication code in the included sample HTTP server to log 
                          important data to the console.
                          23. Updated the included sample HTTP server to implement a fully worked NTLM authentication 
                          example.
                          24. Updated the included sample HTTP server to implement a fully worked Negotiate (i.e. NTLM
                          / Kerberos) authentication example.
                          25. Updated the included sample HTTP server to implement a fully worked Digest (RFC 7616) 
                          authentication example. Please note that the sample code only supports the  digest 
                          authentication logic supported by the most common web browsers (i.e. Internet Explorer,
                          Firefox, Chrome and Edge). What this means is that only MD5 and MD5-sess is supported and 
                          features such as auth-int, SHA-256, SHA-256-sess, SHA-512-256, SHA-512-256-sess, username*, 
                          stale & userhash are not implemented.
         PJN / 22-12-2020 1. Added new CServer::ParseFormVariables methods to aid parsing 
                          application/x-www-form-urlencoded encoded form variables.
                          2. Added new CServer::DecodeUrlEncodedFormValue methods to aid decoding 
                          application/x-www-form-urlencoded encoded form variables.
                          3. Optimized the code in the sample HTTP server to minimize the time a lock is taken on the
                          session cache in various locations.
                          4. Removed the use of the explicit ATL namespace in the sample HTTP server so that the MFC
                          CString classes could be used if so desired.
                          5. Updated the included sample HTTP server to implement a fully worked cookie authentication 
                          example.
         PJN / 23-12-2020 1. Updated the sample app's cookie authentication example to use SHA256 hashing instead of
                          MD5. In addition the password hashing now uses 16 bytes of salt to increase the 
                          cryptographic robustness.
                          2. Added support for new HttpDelegateRequestEx API available in latest Windows 10 SDK.
         PJN / 27-12-2020 1. Added a new CServer::ParseDateHeader method to aid decoding HTTP If-Modified-Since 
                          headers.
                          2. Updated the sample HTTP server to support handling If-Modified-Since headers for the
                          sample file request. These are known as Conditional get requests in the HTTP RFC.
         PJN / 22-01-2021 1. Updated copyright details.
                          2. Reimplemented HandleFileRequest method in the sample app with a new 
                          CServer::SendFileResponse method which provides new reusable functionality to return the 
                          full contents of a specific file in a HTTP response.
         PJN / 15-04-2021 1. Reworked CServer class to be template based. This allows the CppConcurrency::CThreadPool
                          instance to be customized at runtime.
                          2. Fixed a bug in the ParseFormVariables methods where the entity body would not be parsed 
                          correctly if "request.EntityChunkCount is > 1"
         PJN / 16-04-2021 1. CServer::Stop now shuts down the thread pool before it closes the request queue. This
                          reordering of operations can avoid asserts from happening in the thread pool when the 
                          request queue has been closed.
         PJN / 29-06-2021 1. Fixed a bug in SendResponse where an entity chunk would be added to the response even when
                          the "nEntityLength" length parameter was 0. Thanks to David Conalty for reporting this issue.
         PJN / 23-01-2022 1. Updated copyright details.
                          2. Fixed more static code analysis warnings in Visual Studio 2022.
         PJN / 13-02-2022 1. Updated the code to use C++ uniform initialization for all variable declarations
         PJN / 18-05-2023 1. Updated modules to indicate that it needs to be compiled using /std:c++17. Thanks to 
                          Martin Richter for reporting this issue.

Copyright (c) 2020 - 2023 by PJ Naughter (Web: www.naughter.com, Email: pjna@naughter.com)

All rights reserved.

Copyright / Usage Details:

You are allowed to include the source code in any product (commercial, shareware, freeware or otherwise) 
when your product is released in binary form. You are allowed to modify the source code in any way you want 
except you cannot modify the copyright details at the top of each module. If you want to distribute source 
code with your application, then you are only allowed to distribute versions released by the author. This is 
to maintain a single distribution point for the source code.

*/


/////////////////////////// Macros / Defines //////////////////////////////////

#if _MSC_VER > 1000
#pragma once
#endif //#if _MSC_VER > 1000

#if _MSVC_LANG < 201703
#error HTTPServerWrappers requires a minimum C++ language standard of /std:c++17
#endif //#if _MSVC_LANG < 201703

#ifndef __HTTPSERVERWRAPPERS_H__
#define __HTTPSERVERWRAPPERS_H__

#ifndef HTTPSERVERWRAPPERS_EXT_CLASS
#define HTTPSERVERWRAPPERS_EXT_CLASS
#endif //#ifndef HTTPSERVERWRAPPERS_EXT_CLASS

#pragma comment(lib, "httpapi.lib")


/////////////////////////// Includes //////////////////////////////////////////

#ifndef __ATLBASE_H__
#pragma message("To avoid this message, please put atlbase.h in your pre compiled header (normally stdafx.h)")
#include <atlbase.h>
#endif //#ifndef __ATLBASE_H__

#ifndef __ATLSTR_H__
#pragma message("To avoid this message, please put atlstr.h in your pre compiled header (normally stdafx.h)")
#include <atlstr.h>
#endif //#ifndef __ATLSTR_H__

#ifndef __ATLFILE_H__
#pragma message("To avoid this message, please put atlfile.h in your pre compiled header (normally stdafx.h)")
#include <atlfile.h>
#endif //#ifndef __ATLFILE_H__

#ifndef __HTTP_H__
#pragma message("To avoid this message, please put http.h in your pre compiled header (normally stdafx.h)")
#include <http.h>
#endif //#ifndef __HTTP_H__

#ifndef __BCRYPT_H__
#pragma message("To avoid this message, please put bcrypt.h in your pre compiled header (normally stdafx.h)")
#include <bcrypt.h>
#endif //#ifndef __BCRYPT_H__

#ifndef _INC_SHLWAPI
#pragma message("To avoid this message, please put shlwapi.h in your pre compiled header (normally stdafx.h)")
#include <shlwapi.h>
#endif //#ifndef _INC_SHLWAPI

#ifndef _VECTOR_
#pragma message("To avoid this message, please put vector in your pre compiled header (normally stdafx.h)")
#include <vector>
#endif //#ifndef _VECTOR_

#ifndef _ARRAY_
#pragma message("To avoid this message, please put array in your pre compiled header (normally stdafx.h)")
#include <array>
#endif //#ifndef _ARRAY_

#include "CPPCThreadPool.h"


/////////////////////////// Classes ///////////////////////////////////////////

namespace HTTPServer
{

//RAII wrapper for HttpInitialize / HttpTerminate
class HTTPSERVERWRAPPERS_EXT_CLASS CAutoInit
{
public:
//Constructors / Destructors
  CAutoInit() noexcept : m_nFlags{0}
  {
  }
  CAutoInit(const CAutoInit&) = delete;
  CAutoInit(CAutoInit&&) = delete;
  ~CAutoInit() noexcept
  {
    Terminate();
  }

//Methods
  CAutoInit& operator=(const CAutoInit&) = delete;
  CAutoInit& operator=(CAutoInit&&) = delete;

  ULONG Init(ULONG nFlags = HTTP_INITIALIZE_SERVER) noexcept
  {
    m_nFlags = nFlags;
    return HttpInitialize(HTTPAPI_VERSION_2, nFlags, nullptr);
  }

  ULONG Terminate() noexcept
  {
    return HttpTerminate(m_nFlags, nullptr);
  }

protected:
//Member variables
  ULONG m_nFlags;
};


//Class which represents a HTTP Server session
class HTTPSERVERWRAPPERS_EXT_CLASS CSession
{
public:
//Constructors / Destructors
  CSession() noexcept : m_ID{0}
  {
  }
  CSession(const CSession&) = delete;
  CSession(CSession&&) = delete;
  ~CSession() noexcept
  {
    if (m_ID != 0)
      Close();
  }

//Methods
  CSession& operator=(const CSession&) = delete;
  CSession& operator=(CSession&&) = delete;

  _Must_inspect_result_ _Success_(return == NO_ERROR) ULONG Create() noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID == 0);

    return HttpCreateServerSession(HTTPAPI_VERSION_2, &m_ID, 0);
  }

  _Success_(return == NO_ERROR) ULONG Close() noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    const ULONG nReturn{HttpCloseServerSession(m_ID)};
    m_ID = 0;
    return nReturn;
  }

  _Success_(return == NO_ERROR) ULONG SetProperty(IN HTTP_SERVER_PROPERTY Property, _In_reads_bytes_(PropertyInformationLength) PVOID PropertyInformation, IN ULONG PropertyInformationLength) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    return HttpSetServerSessionProperty(m_ID, Property, PropertyInformation, PropertyInformationLength);
  }

  _Success_(return == NO_ERROR) ULONG QueryProperty(IN HTTP_SERVER_PROPERTY Property, _Out_writes_bytes_to_opt_(PropertyInformationLength, *ReturnLength) PVOID PropertyInformation, IN ULONG PropertyInformationLength, _Out_opt_ PULONG ReturnLength) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    return HttpQueryServerSessionProperty(m_ID, Property, PropertyInformation, PropertyInformationLength, ReturnLength);
  }

  operator HTTP_SERVER_SESSION_ID() const noexcept
  {
    return m_ID;
  }

protected:
//Member variables
  HTTP_SERVER_SESSION_ID m_ID;
};

//Class which represents a HTTP Server URL Group
class HTTPSERVERWRAPPERS_EXT_CLASS CURLGroup
{
public:
//Constructors / Destructors
  CURLGroup() noexcept : m_ID{0}
  {
  }
  CURLGroup(const CURLGroup&) = delete;
  CURLGroup(CURLGroup&&) = delete;
  ~CURLGroup() noexcept
  {
    if (m_ID != 0)
      Close();
  }

//Methods
  CURLGroup& operator=(const CURLGroup&) = delete;
  CURLGroup& operator=(CURLGroup&&) = delete;

  _Must_inspect_result_ _Success_(return == NO_ERROR) ULONG Create(_In_ const CSession& session) noexcept
  {
    //Validate our parameters
    ATLASSERT(session.operator HTTP_SERVER_SESSION_ID() != 0);
    ATLASSERT(m_ID == 0);

    return HttpCreateUrlGroup(session, &m_ID, 0);
  }

  _Success_(return == NO_ERROR) ULONG Close() noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    const ULONG nReturn{HttpCloseUrlGroup(m_ID)};
    m_ID = 0;
    return nReturn;
  }

  _Success_(return == NO_ERROR) ULONG AddUrl(IN PCWSTR pFullyQualifiedUrl, IN HTTP_URL_CONTEXT UrlContext OPTIONAL = 0) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    return HttpAddUrlToUrlGroup(m_ID, pFullyQualifiedUrl, UrlContext, 0);
  }

  _Success_(return == NO_ERROR) ULONG RemoveUrl(IN PCWSTR pFullyQualifiedUrl, IN ULONG Flags) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    return HttpRemoveUrlFromUrlGroup(m_ID, pFullyQualifiedUrl, Flags);
  }

  _Success_(return == NO_ERROR) ULONG SetProperty(IN HTTP_SERVER_PROPERTY Property, _In_reads_bytes_(PropertyInformationLength) PVOID PropertyInformation, IN ULONG PropertyInformationLength) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    return HttpSetUrlGroupProperty(m_ID, Property, PropertyInformation, PropertyInformationLength);
  }

  _Success_(return == NO_ERROR) ULONG QueryProperty(IN HTTP_SERVER_PROPERTY Property, _Out_writes_bytes_to_opt_(PropertyInformationLength, *ReturnLength) PVOID PropertyInformation, IN ULONG PropertyInformationLength, _Out_opt_ PULONG ReturnLength) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_ID != 0);

    return HttpQueryUrlGroupProperty(m_ID, Property, PropertyInformation, PropertyInformationLength,ReturnLength);
  }

  operator HTTP_URL_GROUP_ID() const noexcept
  {
    return m_ID;
  }

protected:
//Member variables
  HTTP_URL_GROUP_ID m_ID;
};

//Class which represents a HTTP Server Request Queue
class HTTPSERVERWRAPPERS_EXT_CLASS CRequestQueue
{
public:
//Constructors / Destructors
  CRequestQueue() noexcept : m_Handle{INVALID_HANDLE_VALUE}
  {
  }
  CRequestQueue(const CRequestQueue&) = delete;
  CRequestQueue(CRequestQueue&&) = delete;
  ~CRequestQueue() noexcept
  {
    if (m_Handle != INVALID_HANDLE_VALUE)
      Close();
  }

//Methods
  CRequestQueue& operator=(const CRequestQueue&) = delete;
  CRequestQueue& operator=(CRequestQueue&&) = delete;

  _Must_inspect_result_ _Success_(return == NO_ERROR) ULONG Create(IN PCWSTR Name OPTIONAL = nullptr, IN PSECURITY_ATTRIBUTES SecurityAttributes OPTIONAL = nullptr, IN ULONG Flags OPTIONAL = 0) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle == INVALID_HANDLE_VALUE);

    return HttpCreateRequestQueue(HTTPAPI_VERSION_2, Name, SecurityAttributes, Flags, &m_Handle);
  }

  _Success_(return == NO_ERROR) ULONG Close() noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    const ULONG nReturn{HttpCloseRequestQueue(m_Handle)};
    m_Handle = INVALID_HANDLE_VALUE;
    return nReturn;
  }

  _Success_(return == NO_ERROR) ULONG SetProperty(_In_ HTTP_SERVER_PROPERTY Property, _In_reads_bytes_(PropertyInformationLength) PVOID PropertyInformation, _In_ ULONG PropertyInformationLength) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpSetRequestQueueProperty(m_Handle, Property,PropertyInformation, PropertyInformationLength, 0, nullptr);
  }

  _Success_(return == NO_ERROR) ULONG QueryProperty(_In_ HTTP_SERVER_PROPERTY Property, _Out_writes_bytes_to_opt_(PropertyInformationLength, *ReturnLength) PVOID PropertyInformation, _In_ ULONG PropertyInformationLength, _Out_opt_ PULONG ReturnLength) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpQueryRequestQueueProperty(m_Handle, Property, PropertyInformation, PropertyInformationLength, 0, ReturnLength, nullptr);
  }

  _Success_(return == NO_ERROR) ULONG WaitForDemandStart(IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpWaitForDemandStart(m_Handle, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG Shutdown() noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpShutdownRequestQueue(m_Handle);
  }

  _Success_(return == NO_ERROR) ULONG ReceiveClientCertificate(IN HTTP_CONNECTION_ID ConnectionId, IN ULONG Flags, _Out_writes_bytes_to_(SslClientCertInfoSize, *BytesReceived) PHTTP_SSL_CLIENT_CERT_INFO SslClientCertInfo,
                                                               IN ULONG SslClientCertInfoSize, _Out_opt_ PULONG BytesReceived, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpReceiveClientCertificate(m_Handle, ConnectionId, Flags, SslClientCertInfo, SslClientCertInfoSize,BytesReceived,Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG ReceiveRequest(IN HTTP_REQUEST_ID RequestId, IN ULONG Flags, _Out_writes_bytes_to_(RequestBufferLength, *BytesReturned) PHTTP_REQUEST RequestBuffer,
                                                     IN ULONG RequestBufferLength, _Out_opt_ PULONG BytesReturned, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpReceiveHttpRequest(m_Handle, RequestId, Flags, RequestBuffer, RequestBufferLength, BytesReturned, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG ReceiveRequestEntityBody(IN HTTP_REQUEST_ID RequestId, IN ULONG Flags, _Out_writes_bytes_to_(EntityBufferLength, *BytesReturned) PVOID EntityBuffer,
                                                               IN ULONG EntityBufferLength, _Out_opt_ PULONG BytesReturned, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpReceiveRequestEntityBody(m_Handle, RequestId, Flags, EntityBuffer, EntityBufferLength, BytesReturned, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG SendResponse(IN HTTP_REQUEST_ID RequestId, IN ULONG Flags, IN PHTTP_RESPONSE HttpResponse, IN PHTTP_CACHE_POLICY CachePolicy OPTIONAL,
                                                   OUT PULONG BytesSent OPTIONAL, IN LPOVERLAPPED Overlapped OPTIONAL, IN PHTTP_LOG_DATA LogData OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpSendHttpResponse(m_Handle, RequestId, Flags, HttpResponse, CachePolicy, BytesSent, nullptr, 0, Overlapped, LogData);
  }

  _Success_(return == NO_ERROR) ULONG SendEntityBody(IN HTTP_REQUEST_ID RequestId, IN ULONG Flags, IN USHORT EntityChunkCount OPTIONAL, _In_reads_opt_(EntityChunkCount) PHTTP_DATA_CHUNK EntityChunks,
                                                     OUT PULONG BytesSent OPTIONAL, IN LPOVERLAPPED Overlapped OPTIONAL, IN PHTTP_LOG_DATA LogData OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpSendResponseEntityBody(m_Handle, RequestId, Flags, EntityChunkCount, EntityChunks, BytesSent, nullptr, 0, Overlapped, LogData);
  }

  _Success_(return == NO_ERROR) ULONG DeclarePush(_In_ HTTP_REQUEST_ID RequestId, _In_ HTTP_VERB Verb, _In_ PCWSTR Path, _In_opt_ PCSTR Query, _In_opt_ PHTTP_REQUEST_HEADERS Headers) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpDeclarePush(m_Handle, RequestId, Verb, Path, Query, Headers);
  }

  _Success_(return == NO_ERROR) ULONG WaitForDisconnect(IN HTTP_CONNECTION_ID ConnectionId, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpWaitForDisconnect(m_Handle, ConnectionId, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG WaitForDisconnectEx(IN HTTP_CONNECTION_ID ConnectionId, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpWaitForDisconnectEx(m_Handle, ConnectionId, 0, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG CancelRequest(IN HTTP_REQUEST_ID RequestId, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpCancelHttpRequest(m_Handle, RequestId, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG FlushResponseCache(IN PCWSTR UrlPrefix, IN ULONG Flags, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpFlushResponseCache(m_Handle, UrlPrefix, Flags, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG AddFragmentToCache(IN PCWSTR UrlPrefix, IN PHTTP_DATA_CHUNK DataChunk, IN PHTTP_CACHE_POLICY CachePolicy, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpAddFragmentToCache(m_Handle, UrlPrefix, DataChunk, CachePolicy, Overlapped);
  }

  _Success_(return == NO_ERROR) ULONG ReadFragmentFromCache(IN PCWSTR UrlPrefix, IN PHTTP_BYTE_RANGE ByteRange OPTIONAL, _Out_writes_bytes_to_(BufferLength, *BytesRead) PVOID Buffer,
                                                            IN ULONG BufferLength, _Out_opt_ PULONG BytesRead, IN LPOVERLAPPED Overlapped OPTIONAL) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpReadFragmentFromCache(m_Handle, UrlPrefix, ByteRange, Buffer, BufferLength, BytesRead, Overlapped);
  }

__if_exists(HttpDelegateRequestEx)
{
  _Success_(return == NO_ERROR) ULONG DelegateRequestEx(_In_ HANDLE DelegateQueueHandle, _In_ HTTP_REQUEST_ID RequestId, _In_ HTTP_URL_GROUP_ID DelegateUrlGroupId,
                                                        _In_ ULONG PropertyInfoSetSize, _In_ PHTTP_DELEGATE_REQUEST_PROPERTY_INFO PropertyInfoSet) noexcept
  {
    //Validate our parameters
    ATLASSERT(m_Handle != INVALID_HANDLE_VALUE);

    return HttpDelegateRequestEx(m_Handle, DelegateQueueHandle, RequestId, DelegateUrlGroupId, PropertyInfoSetSize, PropertyInfoSet);
  }
}

  operator HANDLE() noexcept
  {
    return m_Handle;
  }

protected:
//Member variables
  HANDLE m_Handle;
};

//The actual class which you use at runtime to implement a HTTP server in your application
template <typename TThreadPool = CppConcurrency::CThreadPool>
class HTTPSERVERWRAPPERS_EXT_CLASS CServer
{
public:
//Constructors / Destructors
  CServer(_In_ size_t nThreadCount = 0,
          _In_ bool bPermitTaskStealing = true,
          _In_ bool bPaused = false,
          _In_ bool bPumpWindowsMessages = false) noexcept : m_bStopReceiveRequestThread{false},
                                                             m_Threadpool{nThreadCount, bPermitTaskStealing, bPaused, bPumpWindowsMessages},
                                                             m_bBoundedQueue{false},
                                                             m_nMaxQueueSize{0},
                                                             m_nHttpReceiveHttpRequestFlags{0}
  {
  }
  CServer(const CServer&) = delete;
  CServer(CServer&&) = delete;
  virtual ~CServer()
  {
#pragma warning(suppress: 26447)
    Stop();
  }

//Methods
  CServer& operator=(const CServer&) = delete;
  CServer& operator=(CServer&&) = delete;

#pragma warning(suppress: 26429 26485)
/// @brief Starts the HTTP server
/// @param URLs The array of urls in HTTP Server UrlPrefix format to listen on
/// @return a standard HRESULT to indicate success or failure
  virtual HRESULT Start(_In_ const std::vector<std::wstring>& URLs)
  {
    //Create the HTTP session
    if (m_Session.operator HTTP_SERVER_SESSION_ID() != 0)
      return E_FAIL;
    ULONG nError{m_Session.Create()};
    if (nError != NO_ERROR)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nError);

    //Create the URL group
    if (m_URLGroup.operator HTTP_URL_GROUP_ID() != 0)
    {
      m_Session.Close();
      return E_FAIL;
    }
    nError = m_URLGroup.Create(m_Session);
    if (nError != NO_ERROR)
    {
      m_Session.Close();
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nError);
    }

    //Add the URLs to the URL group
    for (const auto& url : URLs)
    {
#pragma warning(suppress: 26481)
      nError = m_URLGroup.AddUrl(url.c_str());
      if (nError != NO_ERROR)
      {
        m_URLGroup.Close();
        m_Session.Close();
        return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nError);
      }
    }

    //Create the request queue
    if (m_Queue.operator HANDLE() != INVALID_HANDLE_VALUE)
    {
      m_URLGroup.Close();
      m_Session.Close();
      return E_FAIL;
    }
    nError = m_Queue.Create();
    if (nError != NO_ERROR)
    {
      m_URLGroup.Close();
      m_Session.Close();
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nError);
    }

    //Associate the URL group with the HTTP request queue
    HTTP_BINDING_INFO bindingInfo{};
    bindingInfo.Flags.Present = TRUE;
    bindingInfo.RequestQueueHandle = m_Queue;
    nError = m_URLGroup.SetProperty(HttpServerBindingProperty, &bindingInfo, sizeof(bindingInfo));
    if (nError != NO_ERROR)
    {
      m_Queue.Close();
      m_URLGroup.Close();
      m_Session.Close();
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nError);
    }

    //Create the read thread
    if (m_ReceiveRequestThread.joinable())
    {
      m_Queue.Close();
      m_URLGroup.Close();
      m_Session.Close();
      return E_FAIL;
    }
    m_bStopReceiveRequestThread = false;
    try
    {
      m_ReceiveRequestThread = std::thread(&CServer::ReceiveRequestThread, this);
    }
    catch (const std::system_error& e)
    {
      m_Queue.Close();
      m_URLGroup.Close();
      m_Session.Close();
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, e.code().value());
    }

    //Start the thread pool
    m_Threadpool.start();

    return S_OK;
  }

/// @brief Stops the HTTP server
/// @return a standard HRESULT to indicate success or failure
  HRESULT Stop()
  {
    //Stop the thread pool
    m_Threadpool.stop();

    //Signal the receive thread to exit before we close down the request queue
    m_bStopReceiveRequestThread = true;

    //Close the request queue (in the process will unblock the blocking call in ReceiveRequestThread worker thread)
    if (m_Queue.operator HANDLE() != INVALID_HANDLE_VALUE)
    {
      m_Queue.Shutdown();
      m_Queue.Close();
    }

    //Wait for the receive thread to stop
    if (m_ReceiveRequestThread.joinable())
      m_ReceiveRequestThread.join();

    //Close down the URL group
    if (m_URLGroup.operator HTTP_URL_GROUP_ID() != 0)
      m_URLGroup.Close();

    //Close down the session
    if (m_Session.operator HTTP_SERVER_SESSION_ID() != 0)
      m_Session.Close();

    return S_OK;
  }

//Member variables
  HTTPServer::CRequestQueue m_Queue;
  HTTPServer::CSession m_Session;
  HTTPServer::CURLGroup m_URLGroup;
  std::thread m_ReceiveRequestThread;
  std::atomic<bool> m_bStopReceiveRequestThread;
  TThreadPool m_Threadpool;
  bool m_bBoundedQueue;
  size_t m_nMaxQueueSize;
  ULONG m_nHttpReceiveHttpRequestFlags;

protected:

//Methods

/// @brief The worker thread which does receives the HTTP requests and dispatches them to the thread pool for handling
  virtual void ReceiveRequestThread()
  {
    //Set the request ID to a null id
    HTTP_REQUEST_ID RequestId;
    HTTP_SET_NULL_ID(&RequestId);

    //Use a default buffer size of 2 KB  + space for the HTTP_REQUEST structure
    ULONG dwRequestBufferLength{sizeof(HTTP_REQUEST) + 2048};

    while (!m_bStopReceiveRequestThread)
    {
      //Allocate the receive buffer
      std::shared_ptr<std::vector<BYTE>> RequestBuffer{std::make_shared<std::vector<BYTE>>(dwRequestBufferLength)};
#pragma warning(suppress: 26429 26490)
      auto pRequest{reinterpret_cast<PHTTP_REQUEST>(RequestBuffer->data())};

      //Empty our the receive buffer
      std::fill(RequestBuffer->begin(), RequestBuffer->end(), 0ui8);

      //Read the HTTP request
      ULONG nBytesRead{0};
#pragma warning(suppress: 26472)
      const ULONG nError{m_Queue.ReceiveRequest(RequestId, m_nHttpReceiveHttpRequestFlags, pRequest, dwRequestBufferLength, &nBytesRead, nullptr)};
      if (m_bStopReceiveRequestThread)
        break;
      if (nError == NO_ERROR)
      {
        //Dispatch the request down to the thread pool to be handled
        if (m_bBoundedQueue)
          m_Threadpool.blocked_submit(m_nMaxQueueSize, &CServer::ProcessRequest, this, std::move(RequestBuffer));
        else
          m_Threadpool.submit(&CServer::ProcessRequest, this, std::move(RequestBuffer));

        //Reset the Request ID to handle the next request
        HTTP_SET_NULL_ID(&RequestId);
      }
      else if (nError == ERROR_MORE_DATA)
      {
        //The input buffer was too small to hold the request headers. Increase the buffer size and call the API again.
        //When calling the API again, handle the request that failed by passing a RequestID. This RequestID is read
        //from the old buffer.
        RequestId = pRequest->RequestId;

        //change the the receive buffer size for the next time around
        dwRequestBufferLength = nBytesRead;
      }
      else if ((nError == ERROR_CONNECTION_INVALID) && !HTTP_IS_NULL_ID(&RequestId))
      {
        //The TCP connection was corrupted by the peer when attempting to handle a request with more buffer.
        //Continue to the next request.
        HTTP_SET_NULL_ID(&RequestId);
      }
    }
  }

/// @brief The main method which handles HTTP requests in the thread pool. By default, this method simply converts the array of bytes representing
/// the HTTP request into a HTTP_REQUEST pointer and calls the HandleRequest method
  virtual void ProcessRequest(_In_ std::shared_ptr<std::vector<BYTE>> request)
  {
    //Convert the raw blob of request data into a HTTP_REQUEST pointer
    //and call the helper method
    auto localRequest{std::move(request)};
#pragma warning(suppress: 26490)
    auto pRequest{reinterpret_cast<PHTTP_REQUEST>(localRequest->data())};
    HandleRequest(pRequest);
  }

/// @brief The method which derived classes can override to handle individual HTTP requests. The default implementation in
/// this class implements a single HTTP GET and returns a 503 for everything else.
#pragma warning(suppress: 26440)
  virtual void HandleRequest(_In_ PHTTP_REQUEST pRequest)
  {
    //Validate our parameters
    ATLASSERT(pRequest != nullptr);

    switch (pRequest->Verb)
    {
      case HttpVerbGET:
      {
        static constexpr const char* pszSampleResponse{R"(<html><head><title>HTTPServerWrappers Demo Page</title></head><p>Your first web server written using HTTPServerWrappers</p></html>)"};
#pragma warning(suppress: 26472)
        SendResponse(pRequest->RequestId, 200, "OK", "text/html", pszSampleResponse, static_cast<ULONG>(strlen(pszSampleResponse)));
        break;
      }
      default:
      {
        SendResponse(pRequest->RequestId, 503, "Not Implemented", nullptr, nullptr, 0);
        break;
      }
    }
  }

/// @brief Sets the HTTP status code and Reason for a HTTP response
  static void InitializeHTTPResponse(_Inout_ HTTP_RESPONSE& response, _In_ USHORT nStatus, _In_z_ PCSTR pszReason) noexcept
  {
    response.StatusCode = nStatus;
    response.pReason = pszReason;
#pragma warning(suppress: 26472)
    response.ReasonLength = static_cast<USHORT>(strlen(pszReason));
  }

/// @brief Adds "Known" headers to a HTTP response
  static void AddKnownHeader(_Inout_ HTTP_RESPONSE& response, _In_ int nHeaderId, _In_z_ PCSTR pszValue) noexcept
  {
#pragma warning(suppress: 26446 26482)
    response.Headers.KnownHeaders[nHeaderId].pRawValue = pszValue;
#pragma warning(suppress: 26446 26472 26482)
    response.Headers.KnownHeaders[nHeaderId].RawValueLength = static_cast<USHORT>(strlen(pszValue));
  }

/// @brief A method to return a HTTP response using just simple function parameters
/// @param RequestID The ID for the request which this response corresponds to.
/// @param nStatusCode The HTTP status code to send e.g. 200, 404 or 503.
/// @param pReason The Reason text to set into "response" e.g. "OK", "File Not Found" or "Not Implemented".
/// @param pContentType The HTTP content - type header to send.
/// @param pEntity The body of the HTTP response to send.
/// @param nEntityLength The length of "pEntity" in bytes.
/// @return a standard HRESULT to indicate success or failure
  HRESULT SendResponse(_In_ HTTP_REQUEST_ID RequestID, _In_ USHORT nStatusCode, _In_z_ PCSTR pReason, _In_opt_z_ PCSTR pContentType, _In_opt_ const void* pEntity, _In_ ULONG nEntityLength) noexcept
  {
    //Initialize the HTTP response structure
    HTTP_RESPONSE response{};

    //Delegate to the other version of SendResponse
    return SendResponse(response, RequestID, nStatusCode, pReason, pContentType, pEntity, nEntityLength);
  }

/// @brief A method to return a HTTP response using an existing HTTP_RESPONSE object and some other simple function parameters
/// @param response The HTTP_RESPONSE struct to use 
/// @param RequestID The ID for the request which this response corresponds to.
/// @param nStatusCode The HTTP status code to send e.g. 200, 404 or 503.
/// @param pReason The Reason text to set into "response" e.g. "OK", "File Not Found" or "Not Implemented".
/// @param pContentType The HTTP content - type header to send.
/// @param pEntity The body of the HTTP response to send.
/// @param nEntityLength The length of "pEntity" in bytes.
/// @return a standard HRESULT to indicate success or failure
  HRESULT SendResponse(_Inout_ HTTP_RESPONSE& response, _In_ HTTP_REQUEST_ID RequestID, _In_ USHORT nStatusCode, _In_z_ PCSTR pReason, _In_opt_z_ PCSTR pContentType, _In_opt_ const void* pEntity, _In_ ULONG nEntityLength) noexcept
  {
    //Initialize the HTTP response structure
    InitializeHTTPResponse(response, nStatusCode, pReason);

    //Add the entity string to the response if one was provided
    HTTP_DATA_CHUNK dataChunk{};
    if ((pEntity != nullptr) && (nEntityLength != 0))
    {
      //Add the content type header
      ATLASSERT(pContentType != nullptr);
#pragma warning(suppress: 6387)
      AddKnownHeader(response, HttpHeaderContentType, pContentType);

      //Add an entity chunk
      dataChunk.DataChunkType = HttpDataChunkFromMemory;
#pragma warning(suppress: 26492)
      dataChunk.FromMemory.pBuffer = const_cast<PVOID>(pEntity);
#pragma warning(suppress: 26472)
      dataChunk.FromMemory.BufferLength = nEntityLength;
      response.EntityChunkCount = 1;
      response.pEntityChunks = &dataChunk;
    }

    //Because the entity body is sent in one call, it is not required to specify the Content-Length
    ULONG nBytesSent{0};
    return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, m_Queue.SendResponse(RequestID, 0, &response, nullptr, &nBytesSent, nullptr, nullptr));
  }

/// @brief A method to return the contents of a file in a HTTP response
/// @param request The request which this response corresponds to.
/// @param pszFile The path of the file to send the response from.
/// @param pszContentType The Content-Type header to use e.g. "text/plain" for this response. Can be left as nullptr if you do not want to specify a content-type header.
/// @param bDoConditionalGet Should the response implement Conditional get semantics for this request.
/// @param pnStatusCode An optional output parameter which will contain the HTTP status code send to the client if a response was successfully generated.
/// @return a standard HRESULT to indicate success or failure
  HRESULT SendFileResponse(_In_ const HTTP_REQUEST request, _In_z_ LPCTSTR pszFile, _In_opt_z_ LPCSTR pszContentType, _In_ bool bDoConditionalGet, _Out_opt_ int* pnStatusCode = nullptr)
  {
    //Open the file to serve the data from
    ATL::CAtlFile file;
    HRESULT hr = file.Create(pszFile, GENERIC_READ, FILE_SHARE_WRITE | FILE_SHARE_READ, OPEN_EXISTING);
    if (FAILED(hr))
    {
      if (pnStatusCode != nullptr)
        *pnStatusCode = 404;
      return SendResponse(request.RequestId, 404, "Not found", nullptr, nullptr, 0);
    }

    //Get the last modified time for the file if required
    bool bHaveFileTime{false};
    SYSTEMTIME stFile{};
    FILETIME ftFile{};
    if (bDoConditionalGet)
    {
      bHaveFileTime = GetFileTime(file, nullptr, nullptr, &ftFile);
      if (bHaveFileTime)
      {
        if (FileTimeToSystemTime(&ftFile, &stFile))
        {
          //Deliberately remove the milliseconds as they do not appear in HTTP responses
          stFile.wMilliseconds = 0;
          bHaveFileTime = SystemTimeToFileTime(&stFile, &ftFile);
        }
      }

      //Get the current system time in UTC
      SYSTEMTIME stCurTime{};
      GetSystemTime(&stCurTime);
      FILETIME ftCurTime{};
      if (SystemTimeToFileTime(&stCurTime, &ftCurTime))
      {
        //Ensure that the file time is not past the server time
        if (bHaveFileTime && (CompareFileTime(&ftFile, &ftCurTime) == 1))
        {
          memcpy_s(&ftFile, sizeof(ftFile), &ftCurTime, sizeof(ftCurTime));
          memcpy_s(&stFile, sizeof(stFile), &stCurTime, sizeof(stCurTime));
        }
      }
    }

    //Start forming the response
    HTTP_RESPONSE response{};
    CStringA sLastModified; //Note it is important that this variable have function level scope to ensure that it lives for
                            //the lifetime of this method
    if (bHaveFileTime)
    {
      if (!GenerateDateHeader(stFile, sLastModified))
      {
        if (pnStatusCode != nullptr)
          *pnStatusCode = 500;
        return SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
      }
      AddKnownHeader(response, HttpHeaderLastModified, sLastModified);
      AddKnownHeader(response, HttpHeaderExpires, sLastModified);
    }

    //Handle conditional GET of the file
    CStringA sIfModifiedSince{request.Headers.KnownHeaders[HttpHeaderIfModifiedSince].pRawValue, request.Headers.KnownHeaders[HttpHeaderIfModifiedSince].RawValueLength};
    bool bHaveIfModifiedSince{false};
    SYSTEMTIME stIfModifiedSince{};
    FILETIME ftIfModifiedSince{};
    if (sIfModifiedSince.GetLength())
      bHaveIfModifiedSince = ParseDateHeader(sIfModifiedSince, stIfModifiedSince);
    if (bHaveFileTime && bHaveIfModifiedSince && SystemTimeToFileTime(&stIfModifiedSince, &ftIfModifiedSince) && CompareFileTime(&ftFile, &ftIfModifiedSince) != 1)
    {
      if (pnStatusCode != nullptr)
        *pnStatusCode = 304;
      return SendResponse(response, request.RequestId, 304, "Not Modified", nullptr, nullptr, 0);
    }

    //Get the length of the file
    ULONGLONG nFileLength{0};
    hr = file.GetSize(nFileLength);
    if (FAILED(hr))
    {
      if (pnStatusCode != nullptr)
        *pnStatusCode = 500;
      return SendResponse(request.RequestId, 500, "Internal Server Error", nullptr, nullptr, 0);
    }

    //Initialize the HTTP response structure
    InitializeHTTPResponse(response, 200, "OK");

    //Because the response is sent over multiple API calls, add a content-length
    CStringA sContentLength;
    sContentLength.Format("%I64u", nFileLength);
    AddKnownHeader(response, HttpHeaderContentLength, sContentLength.GetString());

    //Add a Content-Type header if required
    if (pszContentType != nullptr)
      AddKnownHeader(response, HttpHeaderContentType, pszContentType);

    //Send the HTTP response headers
    ULONG nBytesSent{0};
    const ULONG nResult{m_Queue.SendResponse(request.RequestId, HTTP_SEND_RESPONSE_FLAG_MORE_DATA, &response, nullptr, &nBytesSent, nullptr, nullptr)};
    if (nResult != NO_ERROR)
      return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);

    //Finally, send the entity body from a file handle
    HTTP_DATA_CHUNK dataChunk{};
    dataChunk.DataChunkType = HttpDataChunkFromFileHandle;
    dataChunk.FromFileHandle.ByteRange.StartingOffset.QuadPart = 0;
    dataChunk.FromFileHandle.ByteRange.Length.QuadPart = HTTP_BYTE_RANGE_TO_EOF;
    dataChunk.FromFileHandle.FileHandle = file;
    if (pnStatusCode != nullptr)
      *pnStatusCode = 200;
    return m_Queue.SendEntityBody(request.RequestId, 0, 1, &dataChunk, nullptr, nullptr, nullptr);
  }

/// @brief Breaks a string into two parts separated by an equals character
/// @param sToken the string to split
/// @param sKey upon successful return from this method the part of the string before the equals
/// @param sValue upon successful return from this method the part of the string after the equals
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool EqualsSplit(_In_ const CStringA& sToken, _Inout_ CStringA& sKey, _Inout_ CStringA& sValue)
  {
    const int nSeparator{sToken.Find("=")};
    if (nSeparator != -1)
    {
      sKey = sToken.Left(nSeparator);
      sValue = sToken.Mid(nSeparator + 1);
      return true;
    }
    else
      return false;
  }

/// @brief Checks if a specified position in a string is in the middle of a quoted string
/// @param sValue the string to check
/// @param nIndex the index in the string to check
/// @return true if the specified position is in the middle of a quoted string otherwise false
  _Success_(return != false) static bool IsInQuotedString(_In_ const CStringA& sValue, _In_ int nIndex)
  {
    const int nLength{sValue.GetLength()};
    bool bIsInQuote{false};
    for (int i=0; i<nLength; i++)
    {
      if (!bIsInQuote && (i < (nLength-2)) && sValue[i] == '"')
        bIsInQuote = true;
      else if (bIsInQuote && (sValue[i] == '"') && (sValue[i-1] != '\\'))
        bIsInQuote = false;
      if (i == nIndex)
        return bIsInQuote;
    }
    return false;
  }

/// @brief Parses a HTTP header observing quoted-string semantics. The logic in this method is modelled on CStringT::Tokenize.
/// @param sHeader The HTTP header to parse from
/// @param cToken The token separator to use
/// @param iStart The current starting position to tokenize from
/// @return the parsed token. Will be empty if parsing is finished.
  static CStringA HeaderTokenize(_In_ const CStringA& sHeader, _In_ char cToken, _Inout_ int& iStart)
  {
    //Validate our parameters
    ATLASSERT(iStart >= 0);
    if (iStart < 0)
      AtlThrow(E_INVALIDARG);

    if (cToken == 0)
    {
      if (iStart < sHeader.GetLength())
#pragma warning(suppress: 26481)
        return {sHeader.GetString() + iStart};
    }
    else
    {
#pragma warning(suppress: 26481)
      LPCSTR pszPlace{sHeader.GetString() + iStart};
#pragma warning(suppress: 26481)
      LPCSTR pszEnd{sHeader.GetString() + sHeader.GetLength()};
      if (pszPlace < pszEnd)
      {
        //Find the index of the first character starting from "pszPlace" which is not "cToken"
        int nIncluding{-1};
#pragma warning(suppress: 26429 26481)
        for (LPCSTR pszTemp = pszPlace; (nIncluding == -1) && (pszTemp < pszEnd); ++pszTemp)
        {
          if (*pszTemp != cToken)
#pragma warning(suppress: 26472)
            nIncluding = static_cast<int>(pszTemp - pszPlace);
        }
        if (nIncluding == -1)
          nIncluding = 0;

#pragma warning(suppress: 26481)
        if ((pszPlace + nIncluding) < pszEnd)
        {
#pragma warning(suppress: 26481)
          pszPlace += nIncluding;

          //Find the index of the first character starting from "pszPlace" which is "cToken" and 
          //is not in the middle of a quoted string
          int nExcluding{-1};
#pragma warning(suppress: 26429 26481)
          for (LPCSTR pszTemp = pszPlace; (nExcluding == -1) && (pszTemp < pszEnd); ++pszTemp)
          {
#pragma warning(suppress: 26472)
            if ((*pszTemp == cToken) && !IsInQuotedString(sHeader, static_cast<int>(pszTemp - sHeader.GetString())))
#pragma warning(suppress: 26472)
              nExcluding = static_cast<int>(pszTemp - pszPlace);
          }
          if (nExcluding == -1)
#pragma warning(suppress: 26472)
            nExcluding = static_cast<int>(pszEnd - pszPlace);

          const int iFrom{iStart + nIncluding};
          const int nUntil{nExcluding};
          iStart = iFrom + nUntil + 1;
          return sHeader.Mid(iFrom, nUntil);
        }
      }
    }

    //return empty string, done tokenizing
    iStart = -1;
    return {};
  }

/// @brief functor used as comparison for std::map usage using CStringA
  struct CCharCompare
  {
    bool operator()(_In_ const CStringA& a, _In_ const CStringA& b) const
    {
      return a.Compare(b) < 0;
    }
  };

/// @brief functor used as comparison for std::map usage using CStringW
  struct CWCharCompare
  {
    bool operator()(_In_ const CStringW& a, _In_ const CStringW& b) const
    {
      return a.Compare(b) < 0;
    }
  };

#ifdef _UNICODE
  using CTCharCompare = CWCharCompare;
#else
  using CTCharCompare = CCharCompare;
#endif //#ifdef _UNICODE

/// @brief A method which decodes application/x-www-form-urlencoded encoded form variables into a std::map of string key value pairs
/// @param request the HTTP request to parse the form variables from
/// @param keyValues upon successful return from this method, this will contain the parsed key value pairs
/// @return a standard HRESULT to indicate if the form variables were decoded successfully
  HRESULT ParseFormVariables(_In_ const HTTP_REQUEST& request, _Inout_ std::map<CStringA, CStringA, CCharCompare>& keyValues)
  {
    //Empty out the output parameter
    keyValues.clear();

    //Read the entity body from the request
    CStringA sEntityBody;
    if (request.Flags & HTTP_REQUEST_FLAG_MORE_ENTITY_BODY_EXISTS)
    {
      //Allocate space for reading the entity buffer
      constexpr ULONG nEntityBufferLength{2048};
      std::vector<char> entityBuffer{nEntityBufferLength, std::allocator<char>{}};
      auto pEntityBuffer{entityBuffer.data()};

      //Read all the entity body from the HTTP request
      bool bContinue{true};
      while (bContinue)
      {
        //Read the entity chunk from the request
        ULONG nBytesRead{0};
        const ULONG nResult = m_Queue.ReceiveRequestEntityBody(request.RequestId, 0, pEntityBuffer, nEntityBufferLength, &nBytesRead, nullptr);
        switch (nResult)
        {
          case NO_ERROR:
          {
            if (nBytesRead != 0)
              sEntityBody += CStringA(pEntityBuffer, nBytesRead);
            break;
          }
          case ERROR_HANDLE_EOF:
          {
            if (nBytesRead != 0)
              sEntityBody += CStringA(pEntityBuffer, nBytesRead);
            bContinue = false;
            break;
          }
          default:
          {
            return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);
            break;
          }
        }
      }
    }
    else
    {
      if (request.pEntityChunks != nullptr)
      {
        if (request.pEntityChunks->DataChunkType == HttpDataChunkFromMemory)
        {
          for (USHORT i=0; i<request.EntityChunkCount; i++)
#pragma warning(suppress: 26481)
            sEntityBody += CStringA(static_cast<const char*>(request.pEntityChunks[i].FromMemory.pBuffer), request.pEntityChunks[i].FromMemory.BufferLength);
        }
      }
    }

    //parse the entity body into key value pairs
    int nTokenPosition{0};
    CStringA sToken{sEntityBody.Tokenize("&", nTokenPosition)};
    while (sToken.GetLength())
    {
      CStringA sKey;
      CStringA sValue;
      EqualsSplit(sToken, sKey, sValue);
      HRESULT hr{DecodeUrlEncodedFormValue(sKey)};
      if (FAILED(hr))
        return hr;
      hr = DecodeUrlEncodedFormValue(sValue);
      if (FAILED(hr))
        return hr;
      keyValues[sKey] = sValue;

      sToken = sEntityBody.Tokenize(";", nTokenPosition);
    }

    return S_OK;
  }

/// @brief A method which decodes application/x-www-form-urlencoded encoded form variables into a std::map of string key value pairs
/// @param request the HTTP request to parse the form variables from
/// @param keyValues upon successful return from this method, this will contain the parsed key value pairs
/// @return a standard HRESULT to indicate if the form variables were decoded successfully
  HRESULT ParseFormVariables(_In_ const HTTP_REQUEST& request, _Inout_ std::map<CStringW, CStringW, CWCharCompare>& keyValues)
  {
    //Empty out the output parameter
    keyValues.clear();

    //Read the entity body from the request
    CStringA sEntityBody;
    if (request.Flags & HTTP_REQUEST_FLAG_MORE_ENTITY_BODY_EXISTS)
    {
      //Allocate space for reading the entity buffer
      constexpr ULONG nEntityBufferLength{2048};
      std::vector<char> entityBuffer{nEntityBufferLength, std::allocator<char>{}};
      auto pEntityBuffer{entityBuffer.data()};

      //Read all the entity body from the HTTP request
      bool bContinue{true};
      while (bContinue)
      {
        //Read the entity chunk from the request
        ULONG nBytesRead{0};
        const ULONG nResult{m_Queue.ReceiveRequestEntityBody(request.RequestId, 0, pEntityBuffer, nEntityBufferLength, &nBytesRead, nullptr)};
        switch (nResult)
        {
          case NO_ERROR:
          {
            if (nBytesRead != 0)
              sEntityBody += CStringA(pEntityBuffer, nBytesRead);
            break;
          }
          case ERROR_HANDLE_EOF:
          {
            if (nBytesRead != 0)
              sEntityBody += CStringA(pEntityBuffer, nBytesRead);
            bContinue = false;
            break;
          }
          default:
          {
            return MAKE_HRESULT(SEVERITY_ERROR, FACILITY_WIN32, nResult);
            break;
          }
        }
      }
    }
    else
    {
      if (request.pEntityChunks != nullptr)
      {
        if (request.pEntityChunks->DataChunkType == HttpDataChunkFromMemory)
        {
          for (USHORT i=0; i<request.EntityChunkCount; i++)
#pragma warning(suppress: 26472 26481)
            sEntityBody += CStringA{static_cast<const char*>(request.pEntityChunks[i].FromMemory.pBuffer), static_cast<int>(request.pEntityChunks[i].FromMemory.BufferLength)};
        }
      }
    }

    //parse the entity body into key value pairs
    int nTokenPosition{0};
    CStringA sToken{sEntityBody.Tokenize("&", nTokenPosition)};
    while (sToken.GetLength())
    {
      CStringA sUTF8Key;
      CStringA sUTF8Value;
      EqualsSplit(sToken, sUTF8Key, sUTF8Value);
      CStringW sKey(sUTF8Key);
      HRESULT hr{DecodeUrlEncodedFormValue(sKey)};
      if (FAILED(hr))
        return hr;
      CStringW sValue{sUTF8Value};
      hr = DecodeUrlEncodedFormValue(sValue);
      if (FAILED(hr))
        return hr;
      keyValues[sKey] = sValue;

      sToken = sEntityBody.Tokenize(";", nTokenPosition);
    }

    return S_OK;
  }

/// @brief Decodes a x-www-form-urlencoded encoded string
/// @param sValue the string to decode
/// @return a standard HRESULT to indicate if the form variable was decoded successfully
  static HRESULT DecodeUrlEncodedFormValue(_Inout_ CStringA& sValue)
  {
    //The UrlUnescape API does not handle x-www-form-urlencoded values
    //so fix up the value ahead of decoding it
    sValue.Replace("+", " ");

    PSTR pszValue{sValue.GetBuffer()};
    DWORD dwEscaped{2084}; //INTERNET_MAX_URL_LENGTH
    CStringA sEscaped;
    PSTR pszEscaped{sEscaped.GetBuffer(dwEscaped)};
    const HRESULT hr{UrlUnescapeA(pszValue, pszEscaped, &dwEscaped, 0)};
    sEscaped.ReleaseBuffer(dwEscaped);
    sValue.ReleaseBuffer();
    sValue = sEscaped;
    return hr;
  }

/// @brief Decodes a x-www-form-urlencoded encoded string
/// @param sValue the string to decode
/// @return a standard HRESULT to indicate if the form variable was decoded successfully
  static HRESULT DecodeUrlEncodedFormValue(_Inout_ CStringW& sValue)
  {
    //The UrlUnescape API does not handle x-www-form-urlencoded values
    //so fix up the value ahead of decoding it
    sValue.Replace(L"+", L" ");

    PWSTR pszValue{sValue.GetBuffer()};
    DWORD dwEscaped{2084}; //INTERNET_MAX_URL_LENGTH
    CStringW sEscaped;
    PWSTR pszEscaped{sEscaped.GetBuffer(dwEscaped)};
    const HRESULT hr{UrlUnescapeW(pszValue, pszEscaped, &dwEscaped, URL_UNESCAPE_AS_UTF8)};
    sEscaped.ReleaseBuffer(dwEscaped);
    sValue.ReleaseBuffer();
    sValue = sEscaped;
    return hr;
  }

/// @brief removes leading and trailing quotes and quoted-pairs from a string
/// @param sValue the string to remove quotes and quoted-pairs from
  static void UnquoteHeaderValue(_Inout_ CStringA& sValue)
  {
    //Remove any leading quote
    int nLength{sValue.GetLength()};
    if (nLength && sValue[nLength - 1] == '"')
    {
      sValue = sValue.Left(nLength - 1);
      --nLength;
    }

    //Remove any trailing quote
    if (nLength && sValue[0] == '"')
    {
      sValue = sValue.Mid(1);
      --nLength;
    }

    //Also replace any quoted-pairs
    CStringA sNewValue;
#pragma warning(suppress: 26429)
    char* pszNewValue{sNewValue.GetBuffer(nLength)};
    for (int i=0; i<(nLength - 1); i++)
    {
      if (sValue[i] != '\\')
      {
        *pszNewValue = sValue[i];
#pragma warning(suppress: 26481)
        ++pszNewValue;
      }
      else
      {
        *pszNewValue = sValue[i + 1];
#pragma warning(suppress: 26481)
        ++pszNewValue;
        ++i;
      }
    }
    if (nLength)
    {
      *pszNewValue = sValue[nLength-1];
#pragma warning(suppress: 26481)
      ++pszNewValue;
    }
    *pszNewValue = '\0';
    sNewValue.ReleaseBuffer();
    sValue = sNewValue;
  }

/// @brief Parses a specific cookie from the Cookie HTTP header
/// @param sCookiesHeader The cookies header to parse from
/// @param pszCoookieIdentifier The cookie key value to find
/// @param sCookieValue Upon successful return from this method, this will be the value associated with the cookie
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool ParseCookieFromCookieHeader(_In_ const CStringA& sCookiesHeader, _In_z_ const char* pszCookieIdentifier, _Inout_ CStringA& sCookieValue)
  {
    int nTokenPosition{0};
    CStringA sToken{HeaderTokenize(sCookiesHeader, ';', nTokenPosition)};
    CStringA sSID;
    while (sToken.GetLength()) //NOLINT(clang-analyzer-core.CallAndMessage)
    {
      sToken.Trim();
      if (sToken.Find(pszCookieIdentifier) == 0)
      {
        CStringA sKey;
        const bool bSuccess{EqualsSplit(sToken, sKey, sCookieValue)};
        UnquoteHeaderValue(sCookieValue);
        return bSuccess;
      }
      sToken = HeaderTokenize(sCookiesHeader, ';', nTokenPosition);
    }
    return false;
  }

/// @brief Parses the username and password from a Authorization HTTP header
/// @param sAuthorizationHeader The HTTP authorization header to parse from
/// @param sUsername Upon successful return from this method, this will be the base64 decoded username
/// @param sPassword Upon successful return from this method, this will be the base64 decoded password
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool ParseBasicAuthorizationHeader(_In_ const CStringA& sAuthorizationHeader, _Inout_ CStringA& sUsername, _Inout_ CStringA& sPassword)
  {
    if (sAuthorizationHeader.GetLength() == 0)
      return false;
    int nTokenPosition{0};
    CStringA sToken{HeaderTokenize(sAuthorizationHeader, ' ', nTokenPosition)};
    if ((sToken.CompareNoCase("basic") == 0))
    {
      //Move to the base64 encoded data after the text "Basic"
      sToken = HeaderTokenize(sAuthorizationHeader, ' ', nTokenPosition);
      if (sToken.GetLength())
      {
        //Decode the base64 string passed to us
        const int nEncodedLength{sToken.GetLength()};
        const int nDecodedLength{ATL::Base64DecodeGetRequiredLength(nEncodedLength)};
#pragma warning(suppress: 26472)
        std::vector<BYTE> output{static_cast<size_t>(nDecodedLength) + 1, std::allocator<BYTE>{}};
        int nOutputLength{nDecodedLength};
        if (ATL::Base64Decode(sToken, nEncodedLength, output.data(), &nOutputLength))
        {
          //Null terminate the decoded data
#pragma warning(suppress: 26446)
          output[nOutputLength] = '\0';

          //Extract the username and password from the decoded data
          CStringA sOutput{output.data()};
          const int nColon = sOutput.Find(":");
          if (nColon != -1)
          {
            sUsername = sOutput.Left(nColon);
            sPassword = sOutput.Right(sOutput.GetLength() - nColon - 1);
            return true;
          }
        }
      }
    }
    return false;
  }

/// @brief Parses the base64 encoded data from a Authorization HTTP header
/// @param sAuthorizationHeader The HTTP authorization header to parse from
/// @param pszAuthScheme The name of the AuthorizationScheme to look for in the header e.g. "NTLM" or "Negotiate"
/// @param data Upon successful return from this method, this will be the base64 decoded data
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool ParseAuthorizationHeader(_In_ const CStringA& sAuthorizationHeader, _In_z_ const char* pszAuthScheme, std::vector<BYTE>& data)
  {
    if (sAuthorizationHeader.GetLength() == 0)
      return false;
    int nTokenPosition{0};
    CStringA sToken{HeaderTokenize(sAuthorizationHeader, ' ', nTokenPosition)};
    if (sToken.CompareNoCase(pszAuthScheme) == 0)
    {
      //Move to the base64 encoded data after the authScheme
      sToken = HeaderTokenize(sAuthorizationHeader, ' ', nTokenPosition);
      if (sToken.GetLength())
      {
        //Decode the base64 string passed to us
        const int nEncodedLength{sToken.GetLength()};
        const int nDecodedLength{ATL::Base64DecodeGetRequiredLength(nEncodedLength)};
        data.resize(nDecodedLength);
        int nOutputLength{nDecodedLength};
        if (ATL::Base64Decode(sToken, nEncodedLength, data.data(), &nOutputLength))
        {
          data.resize(nOutputLength);
          return true;
        }
      }
    }
    return false;
  }

/// @brief Parses the base64 encoded data from a Authorization Negotiate HTTP header
/// @param sAuthorizationHeader The HTTP authorization header to parse from
/// @param ntlmData Upon successful return from this method, this will be the base64 decoded data
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool ParseNegotiateAuthorizationHeader(_In_ const CStringA& sAuthorizationHeader, std::vector<BYTE>& ntlmData)
  {
    return ParseAuthorizationHeader(sAuthorizationHeader, "Negotiate", ntlmData);
  }

/// @brief Parses the base64 encoded data from a Authorization NTLM header
/// @param sAuthorizationHeader The HTTP authorization header to parse from
/// @param ntlmData Upon successful return from this method, this will be the base64 decoded data
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool ParseNTLMAuthorizationHeader(_In_ const CStringA& sAuthorizationHeader, std::vector<BYTE>& ntlmData)
  {
    return ParseAuthorizationHeader(sAuthorizationHeader, "NTLM", ntlmData);
  }

/// @brief Structure used to hold parsed Digest Authorization header values
  struct CDigestHeaderVariables
  {
  //Member variables
    CStringA m_sUsername;
    CStringA m_sRealm;
    CStringA m_sNonce;
    CStringA m_sUri;
    CStringA m_sAlgorithm;
    CStringA m_sResponse;
    CStringA m_sQop;
    CStringA m_sNc;
    CStringA m_sCnonce;
    CStringA m_sOpaque;
  };

  enum class DigestAlgorithm
  {
    Undefined,
    MD5,
    MD5Sess
    //Note we do not support SHA256, SHA256Sess, SHA512 or SHA512Sess because no major web browsers currently support it
  };

/// @brief Determine the digest algorithm to use
/// @param variables The digest variables to check
/// @return the DigestAlgorithm to use as determined from the contents of "variables"
  static DigestAlgorithm DetermineDigestAlgorithm(_In_ const CDigestHeaderVariables& variables)
  {
    DigestAlgorithm algorithm{DigestAlgorithm::Undefined};
    if ((variables.m_sAlgorithm.CompareNoCase("MD5") == 0) || variables.m_sAlgorithm.IsEmpty())
      algorithm = DigestAlgorithm::MD5;
    else if (variables.m_sAlgorithm.CompareNoCase("MD5-sess") == 0)
      algorithm = DigestAlgorithm::MD5Sess;
    return algorithm;
  }

/// @Brief Generates a string filled with random data
/// @param dwLengthInBytes the number of bytes of random data to generate
/// @return the generated string. The number of characters in the returned string will be dwLengthInBytes * 2.
  static CStringA GenerateRandom(DWORD dwLengthInBytes)
  {
    BCRYPT_ALG_HANDLE hProv{nullptr};
    NTSTATUS status{BCryptOpenAlgorithmProvider(&hProv, BCRYPT_RNG_ALGORITHM, nullptr, 0)};
    if (!BCRYPT_SUCCESS(status))
      return {};
    std::vector<BYTE> random{dwLengthInBytes, std::allocator<BYTE>{}};
#pragma warning(suppress: 26472)
    status = BCryptGenRandom(hProv, random.data(), static_cast<ULONG>(random.size()), 0);
    BCryptCloseAlgorithmProvider(hProv, 0);
    if (!BCRYPT_SUCCESS(status))
      return {};

    //What will be the return value from this method
    CStringA sData;
    DWORD dwString{0};
#pragma warning(suppress: 26472)
    CryptBinaryToStringA(random.data(), static_cast<DWORD>(random.size()), CRYPT_STRING_HEXRAW | CRYPT_STRING_NOCRLF, nullptr, &dwString);
    char* pszString{sData.GetBuffer(dwString)};
#pragma warning(suppress: 26472)
    CryptBinaryToStringA(random.data(), static_cast<DWORD>(random.size()), CRYPT_STRING_HEXRAW | CRYPT_STRING_NOCRLF, pszString, &dwString);
    sData.ReleaseBuffer();
    return sData;
  }

/// @brief Generates the hash of the specified data
/// @param pszAldId the specified hash algorithm to use
/// @param pbyData the data to hash
/// @param dwDataLength the length of "pbyData" in bytes
/// @return the string representation of the hash
  static CStringA Hash(_In_ LPCWSTR pszAlgId, _In_reads_bytes_(dwDataLength) const BYTE* pbyData, DWORD dwDataLength)
  {
    BCRYPT_ALG_HANDLE hProv{nullptr};
    NTSTATUS status{BCryptOpenAlgorithmProvider(&hProv, pszAlgId, nullptr, 0)};
    if (!BCRYPT_SUCCESS(status))
      return {};
    BCRYPT_HASH_HANDLE hHash{nullptr};
    status = BCryptCreateHash(hProv, &hHash, nullptr, 0, nullptr, 0, 0);
    if (!BCRYPT_SUCCESS(status))
    {
      BCryptCloseAlgorithmProvider(hProv, 0);
      return {};
    }
#pragma warning(suppress: 26492)
    status = BCryptHashData(hHash, const_cast<PUCHAR>(pbyData), dwDataLength, 0);
    if (!BCRYPT_SUCCESS(status))
    {
      BCryptDestroyHash(hHash);
      BCryptCloseAlgorithmProvider(hProv, 0);
      return {};
    }

    DWORD dwHashSize{0};
    DWORD dwPropertySize{sizeof(dwHashSize)};
#pragma warning(suppress: 26490)
    status = BCryptGetProperty(hHash, BCRYPT_HASH_LENGTH, reinterpret_cast<PUCHAR>(&dwHashSize), sizeof(dwHashSize), &dwPropertySize, 0);
    if (!BCRYPT_SUCCESS(status))
    {
      BCryptDestroyHash(hHash);
      BCryptCloseAlgorithmProvider(hProv, 0);
      return {};
    }

    std::vector<BYTE> hash{dwHashSize, std::allocator<BYTE>{}};
#pragma warning(suppress: 26472)
    status = BCryptFinishHash(hHash, hash.data(), static_cast<ULONG>(hash.size()), 0);
    if (!BCRYPT_SUCCESS(status))
    {
      BCryptDestroyHash(hHash);
      BCryptCloseAlgorithmProvider(hProv, 0);
      return {};
    }
    BCryptDestroyHash(hHash);
    BCryptCloseAlgorithmProvider(hProv, 0);

    //What will be the return value from this method
    CStringA sHash;
    DWORD dwString{0};
#pragma warning(suppress: 26472)
    CryptBinaryToStringA(hash.data(), static_cast<DWORD>(hash.size()), CRYPT_STRING_HEXRAW | CRYPT_STRING_NOCRLF, nullptr, &dwString);
    char* pszString{sHash.GetBuffer(dwString)};
#pragma warning(suppress: 26472)
    CryptBinaryToStringA(hash.data(), static_cast<DWORD>(hash.size()), CRYPT_STRING_HEXRAW | CRYPT_STRING_NOCRLF, pszString, &dwString);
    sHash.ReleaseBuffer();
    return sHash;
  }

/// @Brief Generates the MD5 hash of the specified data
/// @param sData the data to hash
/// @return the string representation of the MD5 hash
  static CStringA MD5(_In_ const CStringA& sData)
  {
#pragma warning(suppress: 26490)
    return MD5(reinterpret_cast<const BYTE*>(sData.GetString()), sData.GetLength());
  }

/// @Brief Generates the MD5 hash of the specified data
/// @param pbyData the data to hash
/// @param dwDataLength the length of "pbyData" in bytes
/// @return the string representation of the MD5 hash
  static CStringA MD5(_In_reads_bytes_(dwDataLength) const BYTE* pbyData, DWORD dwDataLength)
  {
    return Hash(BCRYPT_MD5_ALGORITHM, pbyData, dwDataLength);
  }

/// @Brief Generate a "nonce" value suitable for Digest authentication
/// @param sETag the contents of the eTag to use to generate the nonce
/// @return the generated string
  static CStringA GenerateDigestNonce(_In_ const CStringA& sETag)
  {
    SYSTEMTIME st{};
    GetSystemTime(&st);
    CStringA sTimestamp;
    sTimestamp.Format("%d%02d%02d%02d%02d%02d%03d", static_cast<int>(st.wYear), static_cast<int>(st.wMonth), static_cast<int>(st.wDay), static_cast<int>(st.wHour), static_cast<int>(st.wMinute), static_cast<int>(st.wSecond), static_cast<int>(st.wMilliseconds));
    CStringA sSecretData{GenerateRandom(16)};
    return MD5(sTimestamp + ":" + sETag + ":" + sSecretData);
  }

/// @Brief Generates the "H(A1)" string for Digest authentication
/// @param algorithm The digest algorithm to use
/// @param variables The digest variables to use
/// @param sUsername The username to use
/// @param sPassword The password to use
/// @return the H(A1) generated string
  static CStringA GenerateDigestHA1(_In_ DigestAlgorithm algorithm, _In_ const CDigestHeaderVariables& variables, _In_ const CStringA& sUsername, _In_ const CStringA& sPassword)
  {
    CStringA sHA1;
    switch (algorithm)
    {
      case DigestAlgorithm::MD5:
      {
        sHA1 = MD5(sUsername + ":" + variables.m_sRealm + ":" + sPassword);
        break;
      }
      case DigestAlgorithm::MD5Sess:
      {
        sHA1 = MD5(MD5(sUsername + ":" + variables.m_sRealm + ":" + sPassword) + ":" + variables.m_sNonce + ":" + variables.m_sCnonce);
        break;
      }
      default:
      {
        ATLASSERT(FALSE);
        break;
      }
    }
    return sHA1;
  }

/// @Brief Generates the "H(A2)" string for Digest authentication
/// @param algorithm The digest algorithm to use
/// @param variables The digest variables to use
/// @param sMethod the HTTP method to use
/// @param sDigestURI The digestURL to use
/// @param pbyEntityBody the entityBody to use
/// @param dwEntityBodyLength the length of "pbyEntityBody" in bytes
/// @return the H(A2) generated string
  static CStringA GenerateDigestHA2(_In_ DigestAlgorithm algorithm, _In_ const CDigestHeaderVariables& variables, _In_ const CStringA& sMethod, _In_reads_bytes_(dwEntityBodyLength) const BYTE* pbyEntityBody, DWORD dwEntityBodyLength)
  {
    CStringA sHA2;
    switch (algorithm)
    {
      case DigestAlgorithm::MD5: //deliberate fallthrough
      case DigestAlgorithm::MD5Sess:
      {
        if ((variables.m_sQop.CompareNoCase("auth") == 0) || variables.m_sQop.IsEmpty())
          sHA2 = MD5(sMethod + ":" + variables.m_sUri);
        else if (variables.m_sQop.CompareNoCase("auth-int") == 0)
          sHA2 = MD5(sMethod + ":" + variables.m_sUri + ":" + MD5(pbyEntityBody, dwEntityBodyLength));
        break;
      }
      default:
      {
        ATLASSERT(FALSE);
        break;
      }
    }
    return sHA2;
  }

/// @Brief Generates "response" for Digest authentication
/// @param algorithm The digest algorithm to use
/// @param variables The digest variables to use
/// @param sHA1 The "H(A1)" value to use
/// @param sHA2 The "H(A2)" value to use
/// @return the digest "response" string
  static CStringA GenerateDigestResponse(_In_ DigestAlgorithm algorithm, _In_ const CDigestHeaderVariables& variables, _In_ const CStringA& sHA1, _In_ const CStringA& sHA2)
  {
    CStringA sResponse;
    switch (algorithm)
    {
      case DigestAlgorithm::MD5: //deliberate fallthrough
      case DigestAlgorithm::MD5Sess:
      {
        if (variables.m_sQop.IsEmpty())
          sResponse = MD5(sHA1 + ":" + variables.m_sNonce + ":" + sHA2);
        else if ((variables.m_sQop.CompareNoCase("auth") == 0) || (variables.m_sQop.CompareNoCase("auth-int") == 0))
          sResponse = MD5(sHA1 + ":" + variables.m_sNonce + ":" + variables.m_sNc + ":" + variables.m_sCnonce + ":" + variables.m_sQop + ":" + sHA2);
        break;
      }
      default:
      {
        ATLASSERT(FALSE);
        break;
      }
    }
    return sResponse;
  }

/// @brief Parses the various elements associated with Digest authentication from a Authorization HTTP header
/// @param sAuthorizationHeader The HTTP authorization header to parse from
/// @param variables Upon successful return from this method, this will contain the decoded digest authentication variables
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) static bool ParseDigestAuthorizationHeader(_In_ const CStringA& sAuthorizationHeader, _Inout_ CDigestHeaderVariables& variables)
  {
    //Default our output parameter
    variables = CDigestHeaderVariables();

    int nTokenPosition{0};
    int nTokenIndex{0};
    bool bSuccess{false};
    CStringA sToken{HeaderTokenize(sAuthorizationHeader, ' ', nTokenPosition)};
    while (sToken.GetLength()) //NOLINT(clang-analyzer-core.CallAndMessage)
    {
      sToken.Trim();

      if (nTokenIndex == 0)
      {
        if (sToken.CompareNoCase("Digest") != 0)
          return false;
        bSuccess = true;
      }
      else
      {
        CStringA sLowercaseToken{sToken};
        sLowercaseToken.MakeLower();
        if (sLowercaseToken.Find("username=") == 0)
        {
          variables.m_sUsername = sToken.Mid(9);
          UnquoteHeaderValue(variables.m_sUsername);
        }
        else if (sLowercaseToken.Find("realm=") == 0)
        {
          variables.m_sRealm = sToken.Mid(6);
          UnquoteHeaderValue(variables.m_sRealm);
        }
        else if (sLowercaseToken.Find("nonce=") == 0)
        {
          variables.m_sNonce = sToken.Mid(6);
          UnquoteHeaderValue(variables.m_sNonce);
        }
        else if (sLowercaseToken.Find("uri=") == 0)
        {
          variables.m_sUri = sToken.Mid(4);
          UnquoteHeaderValue(variables.m_sUri);
        }
        else if (sLowercaseToken.Find("algorithm=") == 0)
        {
          variables.m_sAlgorithm = sToken.Mid(10);
        }
        else if (sLowercaseToken.Find("response=") == 0)
        {
          variables.m_sResponse = sToken.Mid(9);
          UnquoteHeaderValue(variables.m_sResponse);
        }
        else if (sLowercaseToken.Find("qop=") == 0)
        {
          variables.m_sQop = sToken.Mid(4);
          UnquoteHeaderValue(variables.m_sQop);
        }
        else if (sLowercaseToken.Find("nc=") == 0)
        {
          variables.m_sNc = sToken.Mid(3);
        }
        else if (sLowercaseToken.Find("cnonce=") == 0)
        {
          variables.m_sCnonce = sToken.Mid(7);
          UnquoteHeaderValue(variables.m_sCnonce);
        }
        else if (sLowercaseToken.Find("opaque=") == 0)
        {
          variables.m_sOpaque = sToken.Mid(7);
          UnquoteHeaderValue(variables.m_sOpaque);
        }
      }
      ++nTokenIndex;
      sToken = HeaderTokenize(sAuthorizationHeader, ',', nTokenPosition); //we use "," for subsequent token separators
    }
    return bSuccess;
  }

/// @Brief Helper method to parse weekdays from a HTTP date/time header
/// @return true if parsing was successful otherwise false
  _Success_(return != false) bool ParseWeekDay(_In_z_ const char* pszToken, _Out_ WORD& nWeekDay) noexcept
  {
    bool bSuccess{true};
    if (_stricmp(pszToken, "sun") == 0 || _stricmp(pszToken, "sunday") == 0)
      nWeekDay = 0;
    else if (_stricmp(pszToken, "mon") == 0 || _stricmp(pszToken, "monday") == 0)
      nWeekDay = 1;
    else if (_stricmp(pszToken, "tue") == 0 || _stricmp(pszToken, "tuesday") == 0)
      nWeekDay = 2;
    else if (_stricmp(pszToken, "wed") == 0 || _stricmp(pszToken, "wednesday") == 0)
      nWeekDay = 3;
    else if (_stricmp(pszToken, "thu") == 0 || _stricmp(pszToken, "thursday") == 0)
      nWeekDay = 4;
    else if (_stricmp(pszToken, "fri") == 0 || _stricmp(pszToken, "friday") == 0)
      nWeekDay = 5;
    else if (_stricmp(pszToken, "sat") == 0 || _stricmp(pszToken, "saturday") == 0)
      nWeekDay = 6;
    else
      bSuccess = false;
    return bSuccess;
  }

/// @Brief Helper method to parse months from a HTTP date/time header
/// @return true if parsing was successful otherwise false
  _Success_(return != false) static bool ParseMonth(_In_z_ const char* pszToken, _Out_ WORD& nMonth) noexcept
  {
    if (_stricmp(pszToken, "jan") == 0)
      nMonth = 1;
    else if (_stricmp(pszToken, "feb") == 0)
      nMonth = 2;
    else if (_stricmp(pszToken, "mar") == 0)
      nMonth = 3;
    else if (_stricmp(pszToken, "apr") == 0)
      nMonth = 4;
    else if (_stricmp(pszToken, "may") == 0)
      nMonth = 5;
    else if (_stricmp(pszToken, "jun") == 0)
      nMonth = 6;
    else if (_stricmp(pszToken, "jul") == 0)
      nMonth = 7;
    else if (_stricmp(pszToken, "aug") == 0)
      nMonth = 8;
    else if (_stricmp(pszToken, "sep") == 0)
      nMonth = 9;
    else if (_stricmp(pszToken, "oct") == 0)
      nMonth = 10;
    else if (_stricmp(pszToken, "nov") == 0)
      nMonth = 11;
    else if (_stricmp(pszToken, "dec") == 0)
      nMonth = 12;
    else
      return false;
    return true;
  }

/// @Brief Helper method to parse dates from a HTTP date/time header
/// @param sDateTimeHeader The HTTP authorization header to parse from
/// @param st Upon successful return from this method, this will contain the decoded SYSTEMTIME date/time value
/// @return true if the parsing of the header was successful otherwise false
  _Success_(return != false) bool ParseDateHeader(_In_ const CStringA& sDateTimeHeader, _Out_ SYSTEMTIME& st)
  {
    //This method understands RFC 1123/822, RFC 1036/850 and asctime formats. These are all covered
    //in the HTTP/1.1 RFC of 2616

    //What will be the return value from this function (assume the worst)
    bool bSuccess{false};

    //HTTP times never include a millisecond field, so just set it to zero
    st.wMilliseconds = 0;

    const int nLength{sDateTimeHeader.GetLength()};
    if (nLength > 5)
    {
      const char cThirdCharacter{sDateTimeHeader[3]};
      if (cThirdCharacter == ',') //Parsing a RFC 1123 format date
      {
        //First the weekday
        static constexpr const char* pszTokens{", :"};
        int nTokenPosition{0};
        CStringA sToken{sDateTimeHeader.Tokenize(pszTokens, nTokenPosition)};
        if (nTokenPosition == -1)
          return false;
        bSuccess = ParseWeekDay(sToken, st.wDayOfWeek);

        //Then the day of the month
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wDay = static_cast<WORD>(atoi(sToken));

        //Then the month
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
        bSuccess = bSuccess && ParseMonth(sToken, st.wMonth);

        //And the year
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wYear = static_cast<WORD>(atoi(sToken));

        //And the hour
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wHour = static_cast<WORD>(atoi(sToken));

        //And the minute
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wMinute = static_cast<WORD>(atoi(sToken));

        //And the second
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wSecond = static_cast<WORD>(atoi(sToken));
      }
      else if (cThirdCharacter == ' ') //Parsing an asctime format date
      {
        //First the weekday
        static constexpr const char* pszTokens{", :"};
        int nTokenPosition{0};
        CStringA sToken{sDateTimeHeader.Tokenize(pszTokens, nTokenPosition)};
        if (nTokenPosition == -1)
          return false;
        bSuccess = ParseWeekDay(sToken, st.wDayOfWeek);

        //Then the month
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
        bSuccess = bSuccess && ParseMonth(sToken, st.wMonth);

        //Then the day of the month
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wDay = static_cast<WORD>(atoi(sToken));

        //And the hour
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wHour = static_cast<WORD>(atoi(sToken));

        //And the minute
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wMinute = static_cast<WORD>(atoi(sToken));

        //And the second
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wSecond = static_cast<WORD>(atoi(sToken));

        //And the year
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wYear = static_cast<WORD>(atoi(sToken));
      }
      else //Must be a RFC 1036 format date
      {
        //First the weekday
        static constexpr const char* pszTokens{", :-"};
        int nTokenPosition{0};
        CStringA sToken{sDateTimeHeader.Tokenize(pszTokens, nTokenPosition)};
        if (nTokenPosition == -1)
          return false;
        bSuccess = ParseWeekDay(sToken, st.wDayOfWeek);

        //Then the day of the month
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wDay = static_cast<WORD>(atoi(sToken));

        //Then the month
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
        bSuccess = bSuccess && ParseMonth(sToken, st.wMonth);

        //And the year (2 Digits only, so make some intelligent assumptions)
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wYear = static_cast<WORD>(atoi(sToken));
        if (st.wYear < 50)
          st.wYear += 2000;
        else if (st.wYear < 100)
          st.wYear += 1900; 

        //And the hour
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wHour = static_cast<WORD>(atoi(sToken));

        //And the minute
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wMinute = static_cast<WORD>(atoi(sToken));

        //And the second
        sToken = sDateTimeHeader.Tokenize(pszTokens, nTokenPosition);
        if (nTokenPosition == -1)
          return false;
  #pragma warning(suppress: 26472)
        st.wSecond = static_cast<WORD>(atoi(sToken));
      }
    }
    return bSuccess;
  }

/// @Brief Helper method to generate a HTTP date/time header per RFC 2616
/// @param st The SYSTEMTIME to format
/// @param sValue Upon successful return from this method, this will contain the encoded date/time value
/// @return true if generation of the header was successful otherwise false
  _Success_(return != false) bool GenerateDateHeader(_In_ const SYSTEMTIME& st, _Inout_ CStringA& sValue)
  {
    if (st.wDayOfWeek > 6)
      return false;
    constexpr std::array<const char*, 7> weekdays
    { {
      "Sun",
      "Mon",
      "Tue",
      "Wed",
      "Thu",
      "Fri",
      "Sat"
    } };
    if ((st.wMonth > 12) || (st.wMonth < 1))
      return false;
    constexpr std::array<const char*, 12> months
    { {
      "Jan",
      "Feb",
      "Mar",
      "Apr",
      "May",
      "Jun",
      "Jul",
      "Aug",
      "Sep",
      "Oct",
      "Nov",
      "Dec"
    } };
#pragma warning(suppress: 26446 26482)
    sValue.Format("%s, %d %s %d %02d:%02d:%02d GMT", weekdays[st.wDayOfWeek], static_cast<int>(st.wDay), months[st.wMonth - 1], static_cast<int>(st.wYear), static_cast<int>(st.wHour), static_cast<int>(st.wMinute), static_cast<int>(st.wSecond));
    return true;
  }
};

}; //namespace HTTPServer


#endif //#ifndef __HTTPSERVERWRAPPERS_H__
