/*
Module : TemperWrappers.h
Purpose: Defines the interface for a C++ class which encapsulates the Temper USB thermometer using the Windows HID APIs.
History: PJN / 27-12-2015 1. Initial release
         PJN / 12-01-2020 1. Updated copyright details.
                          2. Fixed more Clang-Tidy static code analysis warnings in the code.
         PJN / 11-02-2022 1. Updated copyright details.
                          2. Updated the code to use C++ uniform initialization for all variable declarations

Copyright (c) 2015 - 2022 by PJ Naughter (Web: www.naughter.com, Email: pjna@naughter.com)

All rights reserved.

Copyright / Usage Details:

You are allowed to include the source code in any product (commercial, shareware, freeware or otherwise) 
when your product is released in binary form. You are allowed to modify the source code in any way you want 
except you cannot modify the copyright details at the top of each module. If you want to distribute source 
code with your application, then you are only allowed to distribute versions released by the author. This is 
to maintain a single distribution point for the source code. 

*/


/////////////////////////// Macros / Defines //////////////////////////////////

#ifndef __TEMPERWRAPPERS_H__
#define __TEMPERWRAPPERS_H__

#ifndef CTEMPERWRAPPERS_EXT_CLASS
#define CTEMPERWRAPPERS_EXT_CLASS
#endif //#ifndef CTEMPERWRAPPERS_EXT_CLASS

#pragma comment(lib, "hid.lib")
#pragma comment(lib, "SetupAPI.lib")


/////////////////////////// Includes //////////////////////////////////////////

#include "HIDWrappers.h"

#ifndef _ARRAY_
#pragma message("To avoid this message, please put array in your pre compiled header (normally stdafx.h)")
#include <array>
#endif //#ifndef _ARRAY_


/////////////////////////// Classes ///////////////////////////////////////////

namespace HID
{

//Wrapper for a HID handle
class CTEMPERWRAPPERS_EXT_CLASS CTemperDevice : public HID::CDevice
{
public:
//Constructors / Destructors
  CTemperDevice() = default;
  CTemperDevice(_In_ const CTemperDevice& device) = delete;
  CTemperDevice(_In_ CTemperDevice&& device) = delete;
  ~CTemperDevice() = default;

//Methods
  CTemperDevice& operator=(_In_ const CTemperDevice& device) = delete;
  CTemperDevice& operator=(_In_ CTemperDevice&& device) = delete;

  static bool Enumerate(_Out_ StringArray& deviceNames, _In_ DWORD dwFlags = DIGCF_PRESENT | DIGCF_DEVICEINTERFACE, _In_opt_ LPCTSTR pszEnumerator = nullptr, _In_ USHORT idVendor = 0xC45, _In_ USHORT idProduct = 0x7401)
  {
    return EnumerateDevices(idVendor, idProduct, deviceNames, dwFlags, pszEnumerator);
  }

  _Success_(return != 0) bool WriteOutputReport(_In_reads_bytes_opt_(dwNumberOfBytesToWrite) const BYTE* byReport, _In_ DWORD dwNumberOfBytesToWrite) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(dwNumberOfBytesToWrite <= 8);

    std::array<BYTE, 9> outputReport;
#pragma warning(suppress: 26446)
    outputReport[0] = 0x00; //The Report ID
#pragma warning(suppress: 26446)
    memcpy_s(&(outputReport[1]), sizeof(outputReport) - 1, byReport, dwNumberOfBytesToWrite);
    DWORD dwWritten{0};
#pragma warning(suppress: 26472)
    return (WriteFile(*this, outputReport.data(), static_cast<DWORD>(outputReport.size()), &dwWritten, nullptr) != FALSE);
  }

  _Success_(return != 0) bool ReadInputReport(_Out_writes_bytes_to_opt_(nNumberOfBytesToRead, *lpNumberOfBytesRead) __out_data_source(FILE) BYTE* pbyInputReport, _In_ DWORD nNumberOfBytesToRead, _Out_opt_ LPDWORD lpNumberOfBytesRead = nullptr) noexcept
  {
    return (ReadFile(*this, pbyInputReport, nNumberOfBytesToRead, lpNumberOfBytesRead, nullptr) != FALSE);
  }

  _Success_(return != 0) bool ReadInputReport(_Inout_ std::vector<BYTE>& inputReport, _Out_opt_ LPDWORD lpNumberOfBytesRead = nullptr) noexcept
  {
#pragma warning(suppress: 26472)
    return (ReadFile(*this, inputReport.data(), static_cast<DWORD>(inputReport.size()), lpNumberOfBytesRead, nullptr) != FALSE);
  }

#pragma warning(suppress: 26429)
  static double GetTemperatureFromInputReport(_In_reads_bytes_(dwReportSize) const BYTE* pbyReport, _In_ DWORD dwReportSize, _In_ DWORD dwHighByteIndex = 5, _In_ DWORD dwLowByteIndex = 6) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(dwHighByteIndex < dwReportSize);
#pragma warning(suppress: 26477)
    ATLASSERT(dwLowByteIndex < dwReportSize);
#pragma warning(suppress: 26477)
    ATLASSERT(pbyReport != nullptr);
    UNREFERENCED_PARAMETER(dwReportSize);

#pragma warning(suppress: 26481)
    const int nLowByte{pbyReport[dwLowByteIndex]};
#pragma warning(suppress: 26472 26481)
    const int nHighByte{static_cast<signed char>(pbyReport[dwHighByteIndex])};
    const int nRawReading{(nHighByte << 8) + (nLowByte & 0xFF)};

    return nRawReading * (125.0 / 32000.0);
  }

  static double GetTemperatureFromInputReport(_In_ const std::vector<BYTE>& report, _In_ DWORD dwHighByteIndex = 5, _In_ DWORD dwLowByteIndex = 6) noexcept
  {
#pragma warning(suppress: 26472)
    return GetTemperatureFromInputReport(report.data(), static_cast<DWORD>(report.size()), dwHighByteIndex, dwLowByteIndex);
  }
};

}; //namespace HID

#endif //#ifndef __TEMPERWRAPPERS_H__
