/*
Module : HIDWrappers.h
Purpose: Defines the interface for a C++ class which encapsulate the Windows HID APIs.
History: PJN / 27-12-2015 1. Initial release
         PJN / 28-11-2017 1. Updated copyright details.
                          2. Replaced BOOL throughout the codebase with bool.
                          3. Replaced CString::operator LPC*STR() calls with CString::GetString calls
         PJN / 18-08-2018 1. Updated copyright details.
                          2. Fixed a number of C++ core guidelines compiler warnings. These changes mean 
                          that the code will now only compile on VC 2017 or later.
                          3. Removed code which supported HIDWRAPPERS_MFC_EXTENSIONS preprocessor
         PJN / 06-05-2019 1. Updated copyright details.
                          2. Updated the code to clean compile on VC 2019
         PJN / 26-08-2019 1. Extended the CDevice::Open method to allow the "dwFlagsAndAttributes" parameter 
                          to be passed to the CreateFile API call.
         PJN / 12-01-2020 1. Updated copyright details.
                          2. Fixed more Clang-Tidy static code analysis warnings in the code.
         PJN / 21-02-2021 1. Updated copyright details.
                          2. Reworked CDevice::EnumerateDevices to use std::vector<BYTE> for its internal
                          temporary buffer.
                          3. Reworked CDevice::EnumerateDevices to use unary predicates and lambdas.
                          4. Provided new public CDevice::GetVendorID and GetProductID methods.
         PJN / 11-02-2022 1. Updated copyright details.
                          2. Updated the code to use C++ uniform initialization for all variable declarations
         PJN / 02-04-2023 1. Updated copyright details.
                          2. Added #pragma once to header file

Copyright (c) 2015 - 2023 by PJ Naughter (Web: www.naughter.com, Email: pjna@naughter.com)

All rights reserved.

Copyright / Usage Details:

You are allowed to include the source code in any product (commercial, shareware, freeware or otherwise) 
when your product is released in binary form. You are allowed to modify the source code in any way you want 
except you cannot modify the copyright details at the top of each module. If you want to distribute source 
code with your application, then you are only allowed to distribute versions released by the author. This is 
to maintain a single distribution point for the source code.

*/


/////////////////////////// Macros / Defines //////////////////////////////////

#pragma once

#ifndef __HIDWRAPPERS_H__
#define __HIDWRAPPERS_H__

#ifndef CHIDWRAPPERS_EXT_CLASS
#define CHIDWRAPPERS_EXT_CLASS
#endif //#ifndef CHIDWRAPPERS_EXT_CLASS


#pragma comment(lib, "hid.lib")
#pragma comment(lib, "SetupAPI.lib")


/////////////////////////// Includes //////////////////////////////////////////

#ifndef _HIDSDI_H
#pragma message("To avoid this message, please put hidsdi.h in your pre compiled header (normally stdafx.h)")
#include <hidsdi.h>
#endif //#ifndef _HIDSDI_H

#ifndef _INC_SETUPAPI
#pragma message("To avoid this message, please put SetupAPI.h in your pre compiled header (normally stdafx.h)")
#include <SetupAPI.h>
#endif //#ifndef _INC_SETUPAPI

#ifndef __ATLBASE_H__
#pragma message("To avoid this message, please put atlbase.h in your pre compiled header (normally stdafx.h)")
#include <atlbase.h>
#endif //#ifndef __ATLBASE_H__

#ifndef _STRING_
#pragma message("To avoid this message, please put string in your pre compiled header (normally stdafx.h)")
#include <string>
#endif //#ifndef _STRING_

#ifndef _VECTOR_
#pragma message("To avoid this message, please put vector in your pre compiled header (normally stdafx.h)")
#include <vector>
#endif //#ifndef _VECTOR_


/////////////////////////// Classes ///////////////////////////////////////////

namespace HID
{

//Typedefs
#ifdef _UNICODE
  using String = std::wstring;
#else
  using String = std::string;
#endif
  using StringArray = std::vector<String>;

//Wrapper for a HID handle
class CHIDWRAPPERS_EXT_CLASS CDevice
{
public:
//Constructors / Destructors
  CDevice() noexcept : m_h{INVALID_HANDLE_VALUE}
  {
  }

  CDevice(_In_ const CDevice& device) = delete;

  CDevice(_In_ CDevice&& device) = delete;

  ~CDevice() noexcept
  {
    Close();
  }

//Methods
  CDevice& operator=(_In_ const CDevice& device) = delete;

  CDevice& operator=(_In_ CDevice&& device) = delete;

  operator HANDLE() const noexcept
  {
    return m_h;
  }

  _Success_(return != false) bool Open(_In_ LPCTSTR pszDeviceName, _In_ DWORD dwDesiredAccess = GENERIC_READ | GENERIC_WRITE, _In_ DWORD dwShareMode = FILE_SHARE_READ | FILE_SHARE_WRITE, _In_opt_ LPSECURITY_ATTRIBUTES lpSecurityAttributes = nullptr, _In_ DWORD dwFlagsAndAttributes = 0) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h == INVALID_HANDLE_VALUE);

    m_h = CreateFile(pszDeviceName, dwDesiredAccess, dwShareMode, lpSecurityAttributes, OPEN_EXISTING, dwFlagsAndAttributes, nullptr);
    return (m_h != INVALID_HANDLE_VALUE);
  }

  void Close() noexcept
  {
    if (m_h != INVALID_HANDLE_VALUE)
    {
      CloseHandle(m_h);
      m_h = INVALID_HANDLE_VALUE;
    }
  }

  //Enumerate devices given a Vendor ID (VID)
  _Success_(return != false) static bool EnumerateDevices(_In_ USHORT nVendorID, _Out_ StringArray& deviceNames, _In_ DWORD dwFlags = DIGCF_PRESENT | DIGCF_DEVICEINTERFACE, _In_opt_ LPCTSTR pszEnumerator = nullptr)
  {
    return EnumerateDevices(deviceNames, [nVendorID](_In_z_ const auto& szDevicePath) noexcept
                            {
                              return (GetVendorID(szDevicePath)== nVendorID);
                            }, dwFlags, pszEnumerator);
  }

  //Enumerate devices given a Vendor ID (VID) and Product ID (PID)
  _Success_(return != false) static bool EnumerateDevices(_In_ USHORT nVendorID, _In_ USHORT nProductID, _Out_ StringArray& deviceNames, _In_ DWORD dwFlags = DIGCF_PRESENT | DIGCF_DEVICEINTERFACE, _In_opt_ LPCTSTR pszEnumerator = nullptr)
  {
    return EnumerateDevices(deviceNames, [nVendorID, nProductID](_In_z_ const auto& szDevicePath) noexcept
                            {
                              return (GetVendorID(szDevicePath) == nVendorID) && (GetProductID(szDevicePath) == nProductID);
                            }, dwFlags, pszEnumerator);
  }

  //Enumerate devices using a binary predicate to decide if the device should be included
template <class _Pr>
  _Success_(return != false) static bool EnumerateDevices(_Inout_ StringArray& deviceNames, _Pr _Pred, _In_ DWORD dwFlags = DIGCF_PRESENT | DIGCF_DEVICEINTERFACE, _In_opt_ LPCTSTR pszEnumerator = nullptr)
  {
    //get the GUID of the HID class
    GUID hidGUID{};
    HidD_GetHidGuid(&hidGUID);

    //Enumerate all devices which supports the specified interface GUID
    HDEVINFO hDeviceInfo{SetupDiGetClassDevs(&hidGUID, pszEnumerator, nullptr, dwFlags)};
    if (hDeviceInfo == INVALID_HANDLE_VALUE)
      return false;

    bool bMoreDevices{true};
    DWORD dwDeviceIndex{0};
    while (bMoreDevices)
    {
      //Get the next device which supports the specified interface GUID
      SP_DEVICE_INTERFACE_DATA did{};
      did.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);
      if (!SetupDiEnumDeviceInterfaces(hDeviceInfo, nullptr, &hidGUID, dwDeviceIndex, &did))
      {
        const DWORD dwError{GetLastError()};
        if (dwError != ERROR_NO_MORE_ITEMS)
        {
          SetupDiDestroyDeviceInfoList(hDeviceInfo);
          SetLastError(dwError);
          return false;
        }
        else
          bMoreDevices = false;
      }
      else
      {
        //Get information about the found device
        DWORD dwLength{0};
        if (!SetupDiGetDeviceInterfaceDetail(hDeviceInfo, &did, nullptr, 0, &dwLength, nullptr))
        {
          const DWORD dwError{GetLastError()};
          if (dwError != ERROR_INSUFFICIENT_BUFFER)
          {
            SetupDiDestroyDeviceInfoList(hDeviceInfo);
            SetLastError(dwError);
            return false;
          }
        }
        std::vector<BYTE> DIDD{dwLength, std::allocator<BYTE>{}};
#pragma warning(suppress: 26490)
        auto pDIDD{reinterpret_cast<PSP_DEVICE_INTERFACE_DETAIL_DATA>(DIDD.data())};
        pDIDD->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);
        SP_DEVINFO_DATA devData{};
        devData.cbSize = sizeof(devData);
        if (!SetupDiGetDeviceInterfaceDetail(hDeviceInfo, &did, pDIDD, dwLength, &dwLength, &devData))
        {
          const DWORD dwError{GetLastError()};
          SetupDiDestroyDeviceInfoList(hDeviceInfo);
          SetLastError(dwError);
          return false;
        }

        //Add the device path to the output array if required
#pragma warning(suppress: 26485)
        if (_Pred(pDIDD->DevicePath))
#pragma warning(suppress: 26485 26489)
          deviceNames.emplace_back(pDIDD->DevicePath);

        //Prepare for the next loop
        ++dwDeviceIndex;
      }
    }
    SetupDiDestroyDeviceInfoList(hDeviceInfo);

    return true;
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetAttributes(_Out_ PHIDD_ATTRIBUTES Attributes) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetAttributes(m_h, Attributes);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetPreparsedData(_Out_ _When_(return != 0, __drv_allocatesMem(Mem))  PHIDP_PREPARSED_DATA* PreparsedData) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetPreparsedData(m_h, PreparsedData);
  }

  _Success_(return == TRUE) BOOLEAN FlushQueue() noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_FlushQueue(m_h);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetConfiguration(_Out_writes_bytes_(ConfigurationLength) PHIDD_CONFIGURATION Configuration, _In_ ULONG ConfigurationLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetConfiguration(m_h, Configuration, ConfigurationLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN SetConfiguration(_In_reads_bytes_(ConfigurationLength) PHIDD_CONFIGURATION Configuration, _In_ ULONG ConfigurationLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_SetConfiguration(m_h, Configuration, ConfigurationLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetFeature(_Out_writes_bytes_(ReportBufferLength) PVOID ReportBuffer, _In_ ULONG ReportBufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetFeature(m_h, ReportBuffer, ReportBufferLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN SetFeature(_In_reads_bytes_(ReportBufferLength) PVOID ReportBuffer, _In_ ULONG ReportBufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_SetFeature(m_h, ReportBuffer, ReportBufferLength);
  }

#if (NTDDI_VERSION >= NTDDI_WINXP)
  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetInputReport(_Out_writes_bytes_(ReportBufferLength) PVOID ReportBuffer, _In_ ULONG ReportBufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetInputReport(m_h, ReportBuffer, ReportBufferLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN SetOutputReport(_In_reads_bytes_(ReportBufferLength) PVOID ReportBuffer, _In_ ULONG ReportBufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_SetOutputReport(m_h, ReportBuffer, ReportBufferLength);
  }
#endif //#if (NTDDI_VERSION >= NTDDI_WINXP)

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetNumInputBuffers(_Out_ PULONG NumberBuffers) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetNumInputBuffers(m_h, NumberBuffers);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN SetNumInputBuffers(_In_ ULONG NumberBuffers) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_SetNumInputBuffers(m_h, NumberBuffers);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetPhysicalDescriptor(_Out_writes_bytes_(BufferLength) PVOID Buffer, _In_ ULONG BufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetPhysicalDescriptor(m_h, Buffer, BufferLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetManufacturerString(_Out_writes_bytes_(BufferLength) PVOID Buffer, _In_ ULONG BufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetManufacturerString(m_h, Buffer, BufferLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetProductString(_Out_writes_bytes_(BufferLength) PVOID Buffer, _In_ ULONG BufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetProductString(m_h, Buffer, BufferLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetIndexedString(_In_ ULONG StringIndex, _Out_writes_bytes_(BufferLength) PVOID Buffer, _In_ ULONG BufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetIndexedString(m_h, StringIndex, Buffer, BufferLength);
  }

  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetSerialNumberString(_Out_writes_bytes_(BufferLength) PVOID Buffer, _In_ ULONG BufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetSerialNumberString(m_h, Buffer, BufferLength);
  }

#if (NTDDI_VERSION >= NTDDI_WINXP)
  _Must_inspect_result_ _Success_(return == TRUE) BOOLEAN GetMsGenreDescriptor(_Out_writes_bytes_(BufferLength) PVOID Buffer, _In_ ULONG BufferLength) noexcept
  {
    //Validate our parameters
#pragma warning(suppress: 26477)
    ATLASSERT(m_h != INVALID_HANDLE_VALUE);

    return HidD_GetMsGenreDescriptor(m_h, Buffer, BufferLength);
  }
#endif //#if (NTDDI_VERSION >= NTDDI_WINXP)

//Static methods
  _Must_inspect_result_ static int GetVendorID(_In_z_ LPCTSTR pszDevicePath) noexcept
  {
    //Allocate a non-const copy of "pszDevicePath" suitable for token parsing
    LPTSTR pszLocalDevicePath{_tcsdup(pszDevicePath)};
    if (pszLocalDevicePath == nullptr)
      return -1;

    LPTSTR pszNextToken{nullptr};
    LPTSTR pszToken{_tcstok_s(pszLocalDevicePath, _T("\\#&"), &pszNextToken)};
#pragma warning(suppress: 26489)
    while (pszToken != nullptr)
    {
      if (_tcsnicmp(pszToken, _T("VID_"), 4) == 0) //Match on the vendor ID
      {
#pragma warning(suppress: 26481)
        LPCTSTR pszVendorID{pszToken + 4};
        const int nVendorID{_tcstol(pszVendorID, nullptr, 16)};
#pragma warning(suppress: 26408)
        free(pszLocalDevicePath); //Free up the heap memory we have used
        return nVendorID;
      }

      //Prepare for the next loop
      pszToken = _tcstok_s(nullptr, _T("\\#&"), &pszNextToken);
    }

#pragma warning(suppress: 26408)
    free(pszLocalDevicePath); //Free up the heap memory we have used

    return -1;
  }

  _Must_inspect_result_ static int GetProductID(_In_z_ LPCTSTR pszDevicePath) noexcept
  {
    //Allocate a non-const copy of "pszDevicePath" suitable for token parsing
    LPTSTR pszLocalDevicePath{_tcsdup(pszDevicePath)};
    if (pszLocalDevicePath == nullptr)
      return -1;

    LPTSTR pszNextToken{nullptr};
    LPTSTR pszToken{_tcstok_s(pszLocalDevicePath, _T("\\#&"), &pszNextToken)};
#pragma warning(suppress: 26489)
    while (pszToken != nullptr)
    {
      if (_tcsnicmp(pszToken, _T("PID_"), 4) == 0) //Match on the product ID
      {
#pragma warning(suppress: 26481)
        LPCTSTR pszProductID{pszToken + 4};
        const int nProductID{_tcstol(pszProductID, nullptr, 16)};
#pragma warning(suppress: 26408)
        free(pszLocalDevicePath); //Free up the heap memory we have used
        return nProductID;
      }

      //Prepare for the next loop
      pszToken = _tcstok_s(nullptr, _T("\\#&"), &pszNextToken);
    }

#pragma warning(suppress: 26408)
    free(pszLocalDevicePath); //Free up the heap memory we have used

    return -1;
  }

protected:
//Member variables
  HANDLE m_h;
};

}; //namespace HID

#endif //#ifndef __HIDWRAPPERS_H__
